function [sol, data] = rpp_read_solution(oid, run, lab)
%RPP_READ_SOLUTION   Read 'rpp' solution and toolbox data from disk.
%
% Extract data structure associated with 'rpp' instance identifier and
% construct solution structure using trajectory segments from individual
% 'coll' instances.
%
% [SOL DATA] = RPP_READ_SOLUTION(OID, RUN, LAB)
%
% SOL  - Solution (struct).
% DATA - Toolbox data (struct).
% OID  - Object instance identifier (string).
% RUN  - Run identifier (string).
% LAB  - Solution label (integer).
%
% See also: FLQT_READ_SOLUTION, COLL_READ_SOLUTION

%   % Copyright (C) James Hannam, Bernd Krauskopf, Hinke M. Osigna
%   $Id: rpp_read_solutions.m 2020-11-26 15:50 NZDT$

tbid = coco_get_id(oid, 'rpp');
data = coco_read_solution(tbid, run, lab);

seg = cell(1, data.nsegs);
for i = [ 1, 2 ]
  segoid = coco_get_id(tbid, sprintf('seg%d', i)); % 'coll' object instance identifier
  seg{i} = coll_read_solution(segoid, run, lab);   % Trajectory segment
end

% concatenate solution depending on mode condition
sol   = seg{1};
sol.T = seg{1}.T + seg{2}.T;
cdata = coco_read_solution( coco_get_id( segoid, 'coll' ), run, lab );
tol   = cdata.ode.TOL;
switch data.mode
  case 1  % mode 1 - locate zero-phase point on periodic orbit, or roate
    % periodic orbit through forward-time phase theta in a stabilised
    % (backward-time) system
    % fix solution in case seg2 is one point repeated ntst*(ncol-1) times
    theta = seg{2}.T / sol.T;
    if trivial_theta( theta, tol )  % truncate solution of 'constant' part when seg{i}.T=0
      if data.zpp
        sol.tbp = seg{1}.tbp;
        sol.xbp = seg{1}.xbp;
      else
        sol.tbp = seg{2}.tbp;
        sol.xbp = seg{2}.xbp;
      end
    else
      sol.tbp = [ seg{1}.tbp; seg{1}.T + seg{2}.tbp(2:end) ];
      sol.xbp = [ seg{1}.xbp; seg{2}.xbp(2:end,:) ];
    end
  case 2  % mode 2 - rotate periodic orbit through forward-time phase given
    % by theta
    % fix solution in case seg1 is one point repeated ntst*(ncol-1) times
    theta = seg{1}.T / sol.T;
    if trivial_theta( theta, tol )  % truncate solution of 'constant' part when seg{i}.T=0
      seg{1}.xbp = [];
      seg{1}.tbp = [];
    end
    sol.tbp = [ seg{2}.tbp; seg{2}.T + seg{1}.tbp(2:end) ];
    sol.xbp = [ seg{2}.xbp; seg{1}.xbp(2:end,:) ];
end
sol.theta = theta;
sol       = rmfield(sol, {'u','t'});

end

function flag = trivial_theta( theta, tol )
%TRIVIAL_THETA    Determine if phase on [0,1) is near zero.
% case is taken to account for phases near integer values that correspond
% to near zero phase. In these cases, the solution sol.xbp will likely
% contain a non-monotonous part corresponding to the segment with near zero
% period.

flag = false;  % set flag
if any( abs( [ theta; mod( theta, 1 ); 1 - mod( theta, 1 ) ] ) <= tol )
  flag = true;
end

end
