function prob = rpp_close_segs_v1(prob, data)
%RPP_CLOSE_SEGS   Append an instance of 'rpp' to problem.
%
% PROB = RPP_CLOSE_SEGS(PROB, DATA)
%
% PROB - Continuation problem structure.
% DATA - Toolbox data structure.

%   % Copyright (C) James Hannam, Bernd Krauskopf, Hinke M. Osigna
%   $Id: rpp_close_segs.m 2020-11-26 14:50 NZDT$

zpp     = data.zpp;    % logical
rpp_par = data.rpp_par;
data    = init_data(prob, data);
bc      = data.rpp_bc;
fid     = bc.fid;

% add constraints
nsegs  = data.nsegs;
T_idx  = zeros(nsegs,1);
x0_idx = zeros(nsegs*bc.dim,1);
x1_idx = x0_idx;
s_idx  = cell(1, nsegs);
off = 0;
for i=1:nsegs
  [fdata, uidx] = coco_get_func_data(prob, data.cids{i}, 'data', 'uidx');
  dim  = fdata.xdim;
  maps = fdata.coll_seg.maps;
  T_idx(i)  = uidx(maps.T_idx);
  x0_idx(off + (1:dim)') = uidx(maps.x0_idx);
  x1_idx(off + (1:dim)') = uidx(maps.x1_idx);
  s_idx{i} = uidx(maps.p_idx);
  off  = off + dim;
end
phi_uidx  = [x0_idx; s_idx{1}];
phi_fid   = coco_get_id(fid, 'phi');
xfid      = coco_get_id(fid, 'x0');
pofid     = coco_get_id(fid, 'PO');
% Optional names
tht_pnames = rpp_par(1:3);
x0_pnames  = rpp_par(5:end);
assert(numel(x0_pnames)==dim, ...
    ['%s: number of custom axis variable names supplied with -rpp_par ',...
    'must equal system dimension'], coco_get_id(data.oid, xfid));
UZ_name   = rpp_par{4};
theta_fid = coco_get_id(fid, 'theta');
% add contraints, monitor endpoint and use xchg to make one of the phi
% inactive
prob = coco_add_func(prob, theta_fid, @theta, @theta_DFDU, data, 'inactive',...
    tht_pnames{1}, 'uidx', T_idx);
prob = coco_add_func(prob, phi_fid, @phi, data, 'active',...
    tht_pnames(2:3), 'uidx', phi_uidx);
prob = coco_xchg_pars(prob, tht_pnames{1}, tht_pnames{2+zpp});
% turn phase point event detection on
if ~isempty(data.theta)
  prob = coco_add_event(prob,UZ_name,tht_pnames{1},data.theta);
end
% add start point as continuation paramter
prob = coco_add_pars(prob, xfid, x0_idx((1:dim)+(~zpp)*dim), x0_pnames, 'regular');
% turn event location for maxima in the first coordnate
if zpp
  prob = coco_add_event(prob,'XTRM',tht_pnames{2},0);
end
% two segment periodic boundary conditions
prob = coco_add_glue(prob,pofid,x0_idx,x1_idx([dim+(1:dim)';(1:dim)']));
% Glue redundant copies of problem parameters
for i=2:nsegs
  sfid = coco_get_id(fid, sprintf('shared%d', i-1));
  prob = coco_add_glue(prob, sfid, s_idx{1}, s_idx{i});
end
% Optional monitor functions
if ~isempty(data.pnames)
  pfid = coco_get_id(fid, 'pars');
  prob = coco_add_pars(prob, pfid, s_idx{1}, data.pnames);
end
prob = coco_add_slot(prob, fid, @coco_save_data, data, 'save_full');

end

function data = init_data(prob, data)
%INIT_DATA   Initialize toolbox data for an instance of 'rpp'.

nsegs      = data.nsegs;
fdata      = coco_get_func_data(prob, data.cids{nsegs}, 'data');
bc.dim     = fdata.xdim; % Track total dimension
bc.fhan    = fdata.fhan;
bc.dfdxhan = fdata.dfdxhan;
bc.dfdphan = fdata.dfdphan;

bc.x0_idx = (1:2*bc.dim)';
bc.p_idx  = 2*bc.dim + (1:fdata.pdim)';

bc.fid    = coco_get_id(data.oid, 'rpp');

data.rpp_bc = bc;
data.no_save = [ data.no_save ...
  { 'rpp_bc' } ];

end

function [data, y] = theta(~, data, u)
%THETA   COCO-compatible wrapper to pahse monitoring function.

% T1 = u(1);
% T2 = u(2);

y = u(1) / ( u(1) + u(2) );  % T1 / ( T1 + T2 ) = theta - relative phase around the periodic orbit

end

function [data, J ] = theta_DFDU(~, data, u )
%THETA_DFDU  COCO-compatible wrapper to Jacobian for theta.

% T1 = u(1);
% T2 = u(2);

J =  [ u(2), -u(1) ] / ( u(1) + u(2) ) ^ 2; % [ T2 / ( T1 + T2 ) ^ 2, -T1 / ( T1 + T2 ) ^ 2 ];

end

function [data, y] = phi(~, data, u)
%PHI   COCO-compatible wrapper to phase condition function.

pr  = data.pr;
bc  = pr.rpp_bc;
dim = bc.dim;

x0.seg1 = u(bc.x0_idx(1:dim));
x0.seg2 = u(bc.x0_idx(dim+(1:dim)));
% p       = u(bc.p_idx);

y = [ [1,0] * bc.fhan( x0.seg1, u(bc.p_idx) );    % flow in the first co-ordanate at x0
      [1,0] * bc.fhan( x0.seg2, u(bc.p_idx) ) ];

end

% function [ data, J ] = phi_DFDU( ~, data, u )
% %PHI_DFDU  COCO-compatible wrapper to Jacobian for phi.
%
% pr  = data.pr;
% bc  = pr.rpp_bc;
% dim = bc.dim;
%
% x0.seg1 = u(bc.x0_idx(1:dim));
% x0.seg2 = u(bc.x0_idx(dim+(1:dim)));
% p       = u(bc.p_idx);
%
% jacx.seg1 = bc.dfdxhan( x0.seg1, p );
% jacx.seg2 = bc.dfdxhan( x0.seg2, p );
% jacp.seg1 = bc.dfdphan( x0.seg1, p );
% jacp.seg2 = bc.dfdphan( x0.seg2, p );
%
% J                  = zeros( 2, 2 * dim + numel( p ) );
% J(1,1)             = jacx.seg1(1,1);
% J(2,dim+1)         = jacx.seg2(1,1);
% J(1,(2*dim+1):end) = jacp.seg1(1,:);
% J(2,(2*dim+1):end) = jacp.seg2(1,:);
%
% end
