function prob = iscrn_close(prob, data)
%ISCRN_CLOSE_SEGS   Append an instance of 'iscrn' to problem.
%
% PROB = ISCRN_CLOSE_SEGS(PROB, DATA)
%
% PROB - Continuation problem structure.
% DATA - Toolbox data structure.

%   % Copyright (C) James Hannam, Bernd Krauskopf, Hinke M. Osigna
%   $Id: iscrn_close_segs.m 2020-11-26 16:09 NZDT$

data     = init_data(prob, data);
if data.arc
  arc_data = init_arc_data(data);
end
bc       = data.iscrn_bc;
fid      = bc.fid;

% add constraints
[fdata, uidx] = coco_get_func_data(prob, data.cids, 'data', 'uidx');
dim      = fdata.xdim;
maps     = fdata.coll_seg.maps;
x0_idx   = uidx(maps.x0_idx);
x1_idx   = uidx(maps.x1_idx);
T_idx    = uidx(maps.T_idx);
s_idx    = uidx(maps.p_idx);
fund_fid = coco_get_id(fid, 'fund');
x0_fid   = coco_get_id(fid, 'x0');
T_fid    = coco_get_id(fid, 'T');
updt_fid = coco_get_id(fid, 'update_arc');
arc_fid  = coco_get_id(fid, 'arc');
if data.fund
  % add perpendicular error measuring condition if fundamental domain mode is used
  prob = coco_add_func(prob, fund_fid, @fund_bc, data,...
    'regular', data.fund_par{1:max([1, dim-2])}, 'uidx', x0_idx);
end
% add return map anchoring conditions
if data.psi_mode  % return to phase psi, instead of theta
  [psi_data, psi_uidx] = coco_get_func_data(prob, data.psid, 'data', 'uidx');
  psi_maps = psi_data.coll_seg.maps;
  x0_psi_idx = psi_uidx(psi_maps.x0_idx);
  x1_psi_idx = psi_uidx(psi_maps.x1_idx);
  T_psi_idx  = psi_uidx(psi_maps.T_idx);
  s_psi_idx  = psi_uidx(psi_maps.p_idx);
  x0_psi_fid = coco_get_id(fid, 'x0_psi');
  p_psi_fid  = coco_get_id(fid, 'p_psi');
  T_psi_fid  = coco_get_id(fid, 'T_psi');
  % glue x(T) = x_psi(0)
  prob   = coco_add_glue(prob, x0_psi_fid, x1_idx, x0_psi_idx);
  % glue paramters to parameters of segment 1
  prob   = coco_add_glue(prob, p_psi_fid, s_idx, s_psi_idx);
  % fix the period of return of x_psi
  prob   = coco_add_pars(prob, T_psi_fid, T_psi_idx,...
    data.psi_par, 'inactive');
  % replace x(T) with x_psi(dtheta*T)
  x1_idx = x1_psi_idx;
end
% return to phase theta
prob = coco_add_func(prob, fid, @iscrn_bc, @iscrn_bc_DFDU, data, 'zero',...
  'uidx', x1_idx, 'u0', 0);
tau_idx = coco_get_func_data(prob, fid, 'uidx');    % get index of tau
tau_idx = tau_idx(end);
% add parameters to monitor x0 and tau, tracing the isochron
if ~isfield(data, 'x0_t0')  % check tangent stored
  prob    = coco_add_pars(prob, x0_fid, [ tau_idx; x0_idx ],...
    data.iscrn_par(2:end), 'active');
else  % pass intial tangent and values
  prob = coco_add_pars(prob, x0_fid, data.iscrn_par(2:end),...
    data.x0_x0, data.x0_t0, 'active');
  x0_glu_idx = coco_get_func_data(prob, x0_fid, 'uidx');  % get index of x0 variables
  x0_glu_fid = coco_get_id(x0_fid, 'glu');
  prob = coco_add_glue(prob, x0_glu_fid, [ tau_idx; x0_idx ], x0_glu_idx);
end
% fix the period of return of x0
prob  = coco_add_pars(prob, T_fid, T_idx, data.iscrn_par{1}, 'inactive');
if data.arc
  % add monitor and event detection functions for isochron arc-length
  prob = coco_add_slot(prob, updt_fid, @update_arc_length,...
    arc_data, 'update');  % 'update' signal
  prob = coco_add_func(prob, arc_fid, @arc_length, arc_data,...
    'regular', 'l', 'uidx', x0_idx);  % arclength monitor function
  prob = coco_add_event(prob, data.arc_par{2}, 'boundary',...
    data.arc_par{1}, data.l_max);  % arc-length stop event function ('boundary')
end
% optional system parameter monitor functions
if ~isempty(data.pnames)
  pfid = coco_get_id(fid, 'pars');
  prob = coco_add_pars(prob, pfid, s_idx, data.pnames);
end
prob = coco_add_slot(prob, fid, @coco_save_data, data, 'save_full');

end

function data = init_data(prob, data)
%INIT_DATA   Initialize toolbox data for an instance of 'iscrn'.

fdata   = coco_get_func_data(prob, data.cids, 'data');
bc.dim  = fdata.xdim; % Track total dimension

bc.x1_idx  = transpose(1:bc.dim);
bc.gamma_s = transpose(data.gamma_s);
bc.s       = data.s / (data.tau_max);
bc.s_p     = [data.s(2), -data.s(1)];
if data.psi_mode  % move phase point gamma_psi, change linear bundle to match
  bc.gamma_s = transpose(data.gamma_psi);
  bc.s       = data.w_psi / (data.tau_max);
  bc.s_p     = [data.w_psi(2), -data.w_psi(1)];
end
% bc.tau_max = data.tau_max;
bc.fid     = coco_get_id(data.oid, 'iscrn');
data.iscrn_bc = bc;

if data.fund
  bc           = rmfield(bc, {'x1_idx', 's', 's_p'});
  bc.w_p       = [data.s(2), -data.s(1)] / data.delta_max;  % rescale delta such that delta = 1 when delta_max is reached
  if data.psi_mode  % send original base point for computation of fundamental domina, measuring delta relative to w_theta
    bc.gamma_s = transpose(data.gamma_s);
  end
  data.fund_bc = bc;
end

data.no_save = [ data.no_save, { 'iscrn_bc', 'fund_bc' } ];

end

function [ arc_data ] = init_arc_data( data )
%INIT_ARC_DATA   Initialize shared data for an instance of 'arc_length'.

arc_data.cid     = data.cids;
arc_data.l       = data.l;
arc_data.l_max   = data.l_max;
arc_data.x0_old  = transpose( data.x0 );
arc_data.arc_par = data.arc_par;

arc_data        = coco_func_data( arc_data );  % create a shared data structure for arc-length functions

end

function [ data, y ] = fund_bc( ~, data, u )
%FUND_BC  COCO-compatible wrapper to isochron tracing monitoring function.

pr = data.pr;
bc = pr.fund_bc;

x0 = u - bc.gamma_s;    % x0 translated to a reference frame around gamma_s
% w_p       = bc.w_p;    % perpendicular to linear approximation of isochron

y = bc.w_p * x0;    % delta = ( x0 - gamma_theta ) . w_perp

end

function [ data, y ] = iscrn_bc( ~, data, u )
%ISCRN_BC  COCO-compatible wrapper to isochron tracing monitoring function.

pr = data.pr;
bc = pr.iscrn_bc;

x1  = u(bc.x1_idx) - bc.gamma_s;    % x0 translated to a reference frame around gamma_s
% s   = bc.s;    % linear approximation of isochron
% s_p = bc.s_p;    % perpendicular to linear approximation of isochron
tau = u(end);

y = [ (bc.s)   * x1 - tau;   % ( x1 - gamma_s ) . s = tau
      (bc.s_p) * x1 ];    % ( x1 - gamma_s ) . s_perp = 0

end

function [ data, J ] = iscrn_bc_DFDU( ~, data, ~ )
%ISCRN_BC_DFDU  COCO-compatible wrapper to Jacobian for iscrn_bc.

pr = data.pr;
bc = pr.iscrn_bc;

J = [   bc.s, -1;    % [  s_1,  s_2, ..., -1;
      bc.s_p,  0 ];  %   s_p1, s_p2, ...,  0 ];

end

function [ data, fbc ] = arc_length( ~, data, u )
%ARC_LENGTH  COCO-compatible wrapper to monitoring function for the
% arc-length of the curve in phase space given by the isochron.

fbc = data.l + norm( u - data.x0_old );

end

function data = update_arc_length( prob, data, cseg, varargin )
%ARC_LENGTH  COCO-compatible wrapper to update data.l and data.x0_old for
% use in the monitor function arc_length. This slot function will be called
% at intialisation, and before each continuation step.

cid             = data.cid;
[ cdata, uidx ] = coco_get_func_data( prob, cid, 'data', 'uidx' );  % get indexes needed to find the values of x0
u           = cseg.src_chart.x;  % actual values of previously completed steps trajectory are stored in this chart
x0_idx      = cdata.coll_seg.maps.x0_idx;    % index of x0
x0_new      = u( uidx( x0_idx ) );
data.l      = data.l + norm( x0_new - data.x0_old );
data.x0_old = x0_new;

end
