function [prob, varargout] = ode_isol2rpp(prob, oid, varargin)
%ODE_ISOL2RPP   Start rotation of a periodic orbit by continuation from
% initial guess, until x0 reaches a defined pahse point.
%
% PROB     = ODE_ISOL2RPP(PROB, OID, VARARGIN)
% VARARGIN = { COLL [PNAMES] [THETA] [OPTS] }
% OPTS     = { '-rpp_par' RPP_PAR | '-u' | '-zpp' }
%
% Start a continuation of collections of constrained trajectory segments of
% evolution equations of the form x'=f(x,p), where f is some non-linear function.
%
% On input:
%
% PROB       : Continuation problem structure. This structure is created with
%              COCO_PROB prior to calling COLL_ISOL2IRPP explicitly.
%
%              The problem structure may contain settings defined with
%              COCO_SET, which will influence the behavior of each 'coll'
%              instance embedded in a 'rpp' instance. Execute COLL_SETTINGS
%              on the Matlab command line to see an overview of COLL
%              toolbox settings.
%
% OID        : Object instance identifier (string). The corresponding
%              toolbox instance identifier is coco_get_id(OID, 'rpp'). Pass a
%              non-trivial object identifier if an instance.
%
% COLL       : Argument sequence for construction of a 'coll' instance,
%              excluding PNAMES.
%
% PNAMES     : Cell array of string labels for continuation parameters
%              (optional). These string labels will be mapped onto the
%              problem parameters as lab{1}->p(1), lab{2}->p(2), ... If
%              string labels are given, their number must match the number
%              of problem parameters. If only one problem parameter is
%              present, a string may be given instead of a cell array of
%              strings. Omitting this argument is only relevant for
%              embedded problems. In general, this argument must be given.
%
% THETA      : Array of phases. Solutions to the periodic orbit rotation problem
%              of realtive phases contained in THETA are saved as special points,
%              labeled accordingly to either 'THETA' or a repllacement label
%              determined by the option'-rpp_par'.
%
% OPTS       : as dsecribed below
% -rpp_par  : indicates that you wish to supply custion contiuation
%              variable names as the cell array of strings RPP_PAR = {
%              'theta', 'phi0', 'phith', 'THETA', 'x0_1', ..., 'x0_m' }. The
%              first element defines the string label for the asymptotic phase
%              of isochron for which a Floquet vector is to be computed. The
%              second and elements 'phi0' and 'phith' define the string labels
%              for the conintuation variables for flow in the first coordinate
%              of the vector field at the start and end points of the first
%              segment of hte periodic orbit. During continuation one of these
%              variables will be fixed, to anchor the periodic orbit, while the
%              other will be allowed to vary to allow for rotation of the
%              periodic orbit. The fourth element 'THETA' is a string that
%              labels points during continuation with pahses corresponding to
%              those supplied to ODE_ISOL2RPP in the array THETA, described in
%              the section above. The final m elements define labels for the
%              coordinates of the start point of the periodic orbit.
% -u         : indicates that the object for which this isochron arc should
%              be computed is unstable. Since the two-point boundary
%              constraint was set up for an 'stable' object, ODE_ISOL2ISCRN
%              will reverse the direction of time for functions supplied,
%              and check the intial solution provided satisfies those
%              functions in backward-time.
% -zpp       : indicates the setting up of the two-point boundary value
%              problem to rotate a periodic obrit to satisfy the convention
%              that the zero-pahse point should lie at the maximum in the
%              first coordinate. Note that this setting will automatically
%              run continuation when ODE_ISOL2RPP is called, and returns the
%              output arguments [PROB, BD, IND], where PROB is the continuation
%              problem structure, BD is the bifurcation diagram object
%              associated with theocntinuation run, and IND is the label number
%              of the solution that satisfies the zero-phase convention.
%
% See also: RPP_READ_SOLUTION, ODE_ISOL2FLQT, ODEISOL2COLL

%   % Copyright (C) James Hannam, Bernd Krauskopf, Hinke M. Osigna
%   $Id: ode_isol2rpp.m 2020-11-26 15:42 NZDT$

%% Locate maxima in the first co-ordinate

tbid   = coco_get_id(oid, 'rpp');
nsegs  = 2;    % this continuation run will use two segments
str    = coco_stream(varargin{:});
if isa(str.peek, 'cell')
  assert(numel(str)>1, '%s: rpp only works for single segment periodic orbits', tbid);
  str = str.get;
  str = str{:};
end

[seg2_args, seg1_args, opts] = rpp_init_args(str);

% segment 1
segoid    = coco_get_id(tbid, 'seg1');
cids{1}   = coco_get_id(segoid, 'coll');
prob      = ode_isol2coll(prob, segoid, seg1_args);
% segment 2
segoid    = coco_get_id(tbid, 'seg2');
cids{2}   = coco_get_id(segoid, 'coll');
prob      = ode_isol2coll(prob, segoid, seg2_args);

data.pnames = opts.pnames;
data.theta  = opts.theta;
pnum = [];
for i=1:nsegs
  fdata = coco_get_func_data(prob, cids{i}, 'data');
  pdim  = fdata.pdim;
  assert(isempty(fdata.pnames), ...
    '%s: parameter labels must not be passed to coll', tbid);
  assert(isempty(pnum) || pnum==pdim, '%s: %s', tbid, ...
    'number of parameters must be equal for all segments');
  pnum = pdim;
end
assert(pdim==numel(data.pnames) || isempty(data.pnames), ...
  '%s: incompatible number of elements for ''pnames''', tbid);

data.xdim      = fdata.xdim;
data.zpp       = opts.zpp;
data.rpp_par   = opts.rpp_par;
data.mode      = opts.mode;
data.nsegs     = nsegs;
data           = rpp_init_data(prob, data, oid); % initialize 'rpp' instance data structure
data.cids      = cids;
prob = rpp_close_segs(prob, data);
prob = ode_add_tb_info(prob, oid, 'rpp', 'segs', 'segs', rpp_sol_info());

%% locate zero phase point

if data.zpp
  runid  = data.runid;
  cont_pars = data.rpp_par([1, 2, 4 + (1:data.xdim)]);
  bd     = coco(prob, runid, [], 1, cont_pars, [0,1]);
  labs   = coco_bd_labs(bd, 'XTRM');
  Usol   = cell(numel(labs), 1);
  xmax   = zeros(1, numel(labs));
  segoid = coco_get_id(tbid, 'seg1');
  i = 1;
  for lab = labs
    Usol{i} = coll_read_solution(segoid, runid, lab);
    xmax(i) = Usol{i}.xbp(1,1);
    i = i+1;
  end
  [~, max_ind] = max(xmax);
  varargout    = {bd,labs(max_ind)};
else
  varargout = {};
end

end
