function [prob] = ode_isol2iscrn(prob, oid, varargin)
%ODE_ISOL2ISCRN   Start computation of an arc of an isochron by the
%coninuation of a trajectory constrained by a two-point boundary value
%problem such that its start point defines the time-k*T return map to a
%linear approxmiation of that isochron.
%
% PROB     = ODE_ISOL2ISCRN(PROB, OID, VARARGIN)
% VARARGIN = { COLL [PNAMES] P0 S TAU_MAX [OPTS] }
% OPTS = { '-arc' ARC | '-fund' FUND | '-iscrn_par' ISCRN_PAR | '-psi' PSI_ARGS | '-u' }
%
% Start a continuation of collections of constrained trajectory segments of
% evolution equations of the form x'=f(x,p), where f is some non-linear function.
%
% On input:
%
% PROB       : Continuation problem structure. This structure is created with
%              COCO_PROB prior to calling COLL_ISOL2ISCRN explicitly.
%
%              The problem structure may contain settings defined with
%              COCO_SET, which will influence the behavior of each 'coll'
%              instance embedded in a 'iscrn' instance. Execute COLL_SETTINGS
%              on the Matlab command line to see an overview of COLL
%              toolbox settings.
%
% OID        : Object instance identifier (string). The corresponding
%              toolbox instance identifier is coco_get_id(OID, 'iscrn'). Pass a
%              non-trivial object identifier if an instance.
%
% COLL       : Argument sequence for construction of a 'coll' instance,
%              excluding PNAMES.
%
% PNAMES     : Cell array of string labels for continuation parameters
%              (optional). These string labels will be mapped onto the
%              problem parameters as lab{1}->p(1), lab{2}->p(2), ... If
%              string labels are given, their number must match the number
%              of problem parameters. If only one problem parameter is
%              present, a string may be given instead of a cell array of
%              strings. Omitting this argument is only relevant for
%              embedded problems. In general, this argument must be given.
% P0         : Initial solution guess for problem parameters. Pass an empty
%              array to indicate absence of problem parameters.
% S          : a vector defining the linear approximation of the isochron
%              that we wish to compute. This will be used to define the
%              time-k*T return map.
% TAU_MAX    : the maximum distance along the linear approximation to be
%              spanned by the end-point of trjectory x(t). This can be used
%              to set up a sensible boundary in cases where x(t) should not
%              return to a point outside of the basin of attraction of the
%              object for which the isochron is being computed. Unless
%              option '-fund' is used, this rescales TAU` = TAU / TAU_MAX
%              such that it lies in [0,1].
%
% OPTS       : as dsecribed below
% -arc       : indicates the computation of arc length along isochron. Cell
%              ARC = { L0, L_MAX, [ARC_PAR] } defines the parameters
%              required for this computation; the intial arc length L0 on
%              the isochron arc, a maximum arc length L_MAX at which to
%              teminate conintuation, and optionally cell array of string
%              labels ARC_PAR = { 'l', 'ARC' }. ARC_PAR contains the
%              continuation variable for the arc length, and the identifyer
%              for for when continuation should be terminated due to
%              reaching maximum arc length - generally 2-4 characters.
%              The default string vlaues for ARC_PAR are shown above, and
%              may be changed to avoid clashes with other user defined
%              labels.
% -fund      : indicates the setting up of the two-point boundary value
%              problem to determine a fundamental domain for use in the
%              compuation of subsequent arcs of the isochron. The cell FUND
%              = { DELTA_MAX, MU, FUND_PAR } defines the parameters
%              required for the boundary value problem. The parameter
%              DELTA_MAX defines the maximal orthogonal deviation from the
%              linear approximation of the isochron at which the time-k*T
%              return map should be accepted. This expresses a notion of
%              error control, and constrains the position of the
%              fundamental domain. The parameter MU is used in rescaling
%              the distance DELTA_MAX for sensitive regions of phase space,
%              and should be the floquet multiplier of the periodic orbit
%              (raised to the period of that periodic orbit), scaling
%              occurs such that DELTA will lie on [0,1], while the scaled
%              TAU` = TAU / ( abs(MU) * sqrt(DELTA_MAX) ) should remain of
%              order 1. The cell array of strings FUND_PAR = { 'delta',
%              ['delta_th', 'delta_n1,, .... ,delta_nm'] } sontains the
%              labels for the continuation variables describing the total
%              orthogonal deviation 'delta' from the linear approximation
%              of the isochron, and in the case of isochrons of saddle-type
%              periodic orbits the orthogonal deviation along the
%              two-dimensional manifold of the periodic orbit 'delta_th'
%              (phase shear) and orhogonal deviations along the remaining
%              directions 'delta_n1', ...
% -iscrn_par : indicates that you wish to supply custion contiuation
%              variable names as the cell array of strings ISCRN_PAR = {
%              'T', 'tau', 'x0_1', 'x0_2', [..., 'x0_m'] }. The first
%              element defines the string label for the time of the
%              time-k*T return map for this arc of the isochron. The second
%              element 'tau' defines the string label for the distance
%              along the linear approximation of the isochron of the
%              end-point of the trajectory in the two-point boundary value
%              problem. The final m elements define labels for the
%              coordinates along the isochron arc which are being computed.
% -psi       : indicates the modifcation of the two-point boundary value
%              problem defining the time-k*T return map to the linear
%              approximation of the isochron. Instead, we set up the
%              time-(psi - theta)*T return map to the linear approximation
%              of an isochron of phase psi, in order to compute the
%              fundamental arc of the isochron of phase theta. This option
%              can only be used in conjunction with the option -fund, and
%              contrains the fundamental domain for the isochron for
%              subsequent isochron arc computations using the standard
%              time-k*T return map. The cell PSI_ARGS = { PSI, W_PSI,
%              GAMMA_PSI, ['T_psi'] } defines the parameters required to
%              set this boundary value problem. PSI defines the pahse of
%              the linear approximation W_PSI to be used in the modified
%              return map, and GAMMA_PSI is the base-point of that linear
%              approximation. Teh string 'T_psi' can be chosen to customise
%              the period of the trjectory segment connecting the end-point
%              of the trajectory x(t) to the start point of x_psi(t) such
%              that x(t) would define a time-T return map to the linear
%              approximation for the isochron of pahse theta, and x_psi
%              would define a time-T_psi return map from x(T) to the
%              linear approximation of the isochron of pahse psi.
%              The option '-psi' is supplied for cases where the linear
%              approximation of the isochron which is to be computed lies
%              roughly tangent to periodic orbit, and normal isochron
%              computation would produce misrepreseentative results.
% -u         : indicates that the object for which this isochron arc should
%              be computed is unstable. Since the two-point boundary
%              constraint was set up for an 'stable' object, ODE_ISOL2ISCRN
%              will reverse the direction of time for functions supplied,
%              and check the intial solution provided satisfies those
%              functions in backward-time.
%
% See also: ODE_ISOL2ISCRN, ODEISOL2COLL, ODE_COLL2COLL

%   % Copyright (C) James Hannam, Bernd Krauskopf, Hinke M. Osigna
%   $Id: ode_isol2iscrn.m 2020-11-26 14:53 NZDT$

%% Locate maxima in the first co-ordinate

tbid   = coco_get_id(oid, 'iscrn');
str    = coco_stream(varargin{:});
if isa(str.peek, 'cell')
  assert(numel(str)>1, '%s: iscrn only works for single segment periodic orbits', tbid);
  str = str.get;
  str = str{:};
end

[seg1_args, opts] = iscrn_init_args(str);

% segment 1
cids = coco_get_id(tbid, 'coll');
prob = ode_isol2coll(prob, tbid, seg1_args);
% segment 2 - translation by flow of x(0) by psi from w_th to w_psi
if isfield( opts, 'psi_args' )
  tol = coco_get(prob, 'all');
  tol = tol.TOL;
  % check that psi is distinc from theta
  if any(abs([opts.psi{end}, 1 - opts.psi{end}]) < tol)
    warning('Phase psi is non-distinct from phase of initial solution to computational tolerance, option ''-psi'' ignored.')
    opts.psi = [];
  else
    psid      = coco_get_id('psi', 'iscrn');
    prob      = ode_isol2coll(prob, psid, opts.psi_args);
    data.psid = coco_get_id( psid, 'coll' );  % send to data for iscrn_close
  end
end

data.pnames = opts.pnames;
fdata = coco_get_func_data(prob, cids, 'data');
pdim  = fdata.pdim;
assert(isempty(fdata.pnames), ...
  '%s: parameter labels must not be passed to coll', tbid);
assert(pdim==numel(data.pnames) || isempty(data.pnames), ...
  '%s: incompatible number of elements for ''pnames''', tbid);


data.xdim      = fdata.xdim;
data.s         = opts.s;
data.gamma_s   = opts.gamma_s;
data.tau_max   = opts.tau_max;
data.delta_max = opts.delta_max;
data.iscrn_par = opts.iscrn_par;
data.arc       = opts.arc;
data.psi       = opts.psi;
data           = iscrn_init_data(prob, data, oid); % initialize 'iscrn' instance data structure
data.cids      = cids;
prob = iscrn_close(prob, data);
prob = ode_add_tb_info(prob, oid, 'iscrn', 'seg', 'iscrn', iscrn_sol_info());

end
