function [prob] = ode_isol2flqt(prob, oid, varargin)
%ODE_ISOL2RPP   Start rotation of a periodic orbit by continuation from
% initial guess, until x0 reaches a defined pahse point.
%
% PROB     = ODE_ISOL2FLQT(PROB, OID, VARARGIN)
% VARARGIN = { COLL [PNAMES] [THETA] [OPTS] }
% OPTS     = { '-flqt_par' FLQT_PAR | '-flqt0' FLQT0 | '-u' }
%
% Start a continuation of collections of constrained trajectory segments of
% evolution equations of the form x'=f(x,p), where f is some non-linear function.
%
% On input:
%
% PROB       : Continuation problem structure. This structure is created with
%              COCO_PROB prior to calling COLL_ISOL2IFLQT explicitly.
%
%              The problem structure may contain settings defined with
%              COCO_SET, which will influence the behavior of each 'coll'
%              instance embedded in a 'flqt' instance. Execute COLL_SETTINGS
%              on the Matlab command line to see an overview of COLL
%              toolbox settings.
%
% OID        : Object instance identifier (string). The corresponding
%              toolbox instance identifier is coco_get_id(OID, 'flqt'), or
%              coco_get_id(OID, 'flqt.rpp') when the option '-flqt0' is used.
%              Pass a non-trivial object identifier if an instance.
%
% COLL       : Argument sequence for construction of a 'coll' instance,
%              excluding PNAMES.
%
% PNAMES     : Cell array of string labels for continuation parameters
%              (optional). These string labels will be mapped onto the
%              problem parameters as lab{1}->p(1), lab{2}->p(2), ... If
%              string labels are given, their number must match the number
%              of problem parameters. If only one problem parameter is
%              present, a string may be given instead of a cell array of
%              strings. Omitting this argument is only relevant for
%              embedded problems. In general, this argument must be given.
%
% THETA      : Array of phases, supply when using '-flqt0' option. Solutions to
%              the Floquet vector rotation problem of realtive phases contained
%              in THETA are saved as special points, labeled accordingly to
%              either 'THETA' or a repllacement label determined by the option
%              '-flqt_par'.
%
% OPTS       : as dsecribed below
% -flqt_par  : indicates that you wish to supply custion contiuation
%              variable names as the cell array of strings FLQT_PAR = { RPP_PAR,
%              FLQT_PAR }, where RPP_PAR =  {'theta', 'phi0', 'phith', 'THETA',
%              'gama_1', ..., 'gama_m'},  and FLQT_PAR = {'wth_1', ..., 'wth_m',
%              'mu'}. The first element of the cell RPP_PAR defines the string
%              label for the asymptotic phase of isochron for which a Floquet
%              vector is to be computed. The second and elements 'phi0' and
%              'phith' define the string labels for the conintuation variables
%              for flow in the first coordinate of the vector field at the start
%              and end points of the first segment of hte periodic orbit. During
%              continuation one of these variables will be fixed, to anchor the
%              periodic orbit, while the other will be allowed to vary to allow
%              for rotation of the periodic orbit. The fourth element 'THETA'
%              is a string that labels points during continuation with phases
%              corresponding to those supplied to ODE_ISOL2FLQT in the array
%              THETA, described in the section above. The final m elements define
%              labels for coordinates of the start point of the periodic orbit.
%              The first elements of the cell array FLQT_PAR are strings naming
%              the coordinates of the Floquet vector located that the start
%              on the periodic orbit. There should be m entries, where m is the
%              dimension of the vector field. The final entry in FLQT_PAR is a
%              string label for the Floquet multiplier. 
% -flqt0      : indicates the setting up of the two-point boundary value
%              problem to rotate a periodic obrit with its Floquet vector to
%              desired phases stored in THETA. The cell FLQT0 = { MU, W }
%              defines the parameters required for the boundary value problem.
%              The parameter MU is the Floquet multiplier of the periodic orbit,
%              and the paramter W is the Floquet vector asoociatedd with the
%              start point on the periodic orbit. These paramters may be
%              computed by running continuation on a problem constructed by this
%              construtor without the '-flqt0' option invoked, and read using
%              FLQT_READ_SOLUTION.
% -u         : indicates that the object for which this isochron arc should
%              be computed is unstable. Since the two-point boundary
%              constraint was set up for an 'stable' object, ODE_ISOL2ISCRN
%              will reverse the direction of time for functions supplied,
%              and check the intial solution provided satisfies those
%              functions in backward-time.
%
% See also: FQLT_READ_SOLUTION, ODE_ISOL2RPP, ODEISOL2COLL

%   % Copyright (C) James Hannam, Bernd Krauskopf, Hinke M. Osigna
%   $Id: ode_isol2flqt.m 2020-11-26 14:50 NZDT$

%% Locate maxima in the first co-ordinate

tbid   = coco_get_id(oid, 'flqt');
str    = coco_stream(varargin{:});
if isa(str.peek, 'cell')
  assert(numel(str)>1, '%s: rpp only works for single segment periodic orbits', tbid);
  str = str.get;
  str = str{:};
end

[seg1_args, opts] = flqt_init_args(str);

if isempty(opts.theta)
  nsegs   = 1;    % one segment for floquet bundle location
  segoid  = coco_get_id(tbid, 'seg1');
  cids{1} = coco_get_id(segoid, 'coll');
  prob    = ode_isol2coll(prob, segoid, seg1_args);
else
  nsegs = 2;      % call ode_isol2rpp to set up two segment periodic orbit rotation
  tbid2   = coco_get_id(tbid, 'rpp');    % keep track of cid through isol2rpp
  segoid  = coco_get_id(tbid2, 'seg1');
  cids{1} = coco_get_id(segoid, 'coll');
  segoid  = coco_get_id(tbid2, 'seg2');
  cids{2} = coco_get_id(segoid, 'coll');
  prob    = ode_isol2rpp(prob, tbid, seg1_args);
  data.mu     = opts.flqt0{1};
  data.w0     = opts.flqt0{2};
end

data.pnames = opts.pnames;
data.theta  = opts.theta;
pnum = [];
for i=1:nsegs
  fdata = coco_get_func_data(prob, cids{i}, 'data');
  pdim  = fdata.pdim;
  assert(isempty(fdata.pnames), ...
    '%s: parameter labels must not be passed to coll', tbid);
  assert(isempty(pnum) || pnum==pdim, '%s: %s', tbid, ...
    'number of parameters must be equal for all segments');
  pnum = pdim;
end
assert(pdim==numel(data.pnames) || isempty(data.pnames), ...
  '%s: incompatible number of elements for ''pnames''', tbid);

data.xdim     = fdata.xdim;
data.flqt_par = opts.flqt_par;
data.mode     = opts.mode;
data.nsegs    = nsegs;
data          = flqt_init_data(prob, data, oid); % initialize 'flqt' instance data structure
data.cids     = cids;
if nsegs==1    % set up for two-segment rotation of periodic orbit and floqet bundle is significantly different
  prob  = flqt_close_segs(prob, data);
else
  prob  = flqt_close_segs_theta(prob, data);
end
prob    = ode_add_tb_info(prob, oid, 'flqt', 'segs', 'segs', flqt_sol_info());

end
