function prob = ode_iscrn2iscrn( prob, oid, varargin )
%ODE_ISCRN2ISCRN   Start continuation of trajectory segments from saved solution
% for isochron computation.
%
% PROB = ODE_ISCRN2ISCRN(PROB, OID, VARARGIN)
% VARARGIN = { RUN [SOID] LAB [ORUN [OSOID] [OLAB]] }
%
% Restart a continuation of trajectory segments from a trajectory segment
% that was obtained and saved to disk in a previous continuation. To
% restart from a saved trajectory segment, at least the name RUN of the
% continuation run and the solution label LAB must be given for both the
% terminal solution of the previous isochrona arc, and the fundamental solution,
% the terminal solution of the fundamental domian. This function additionally
% constructs and passes the tangent vector to the continuation problem, to help
% ensure smoothness in the isochron that is computed.
%
% PROB : Continuation problem structure.
% OID  : Target object instance identifier (string). Pass the empty string
%        '' for a simple continuation of trajectory segments.
%
% See ODE_ISOL2COLL for more details on PROB and OID.
%
% RUN   : Run identifier (string or cell-array of strings). Name of the run
%         from which to restart a new continuation run.
% SOID  : Source object instance identifier (string, optional). If the
%         argument SOID is omitted, OID is used. Pass the empty string ''
%         for OID and omit SOID for a simple continuation of trajectory
%         segments. Pass non-trivial object identifiers if an instance of
%         the COLL toolbox is part of a composite continuation problem.
% LAB   : Solution label (integer). The integer label assigned by COCO to an
%         trajectory segment during the continuation run RUN.
% ORUN  : Fundamental domain Run identifier (string). Name of the run
%         from which to extract the fundamental solution for concatenation.
% OSOID : Fundamental domain source object instance identifier (string,
%         optional). If the argument SOID is omitted, OID is used. Pass
%         non-trivial object identifiers, especially when cinstructing composite
%         continuation problem.
% OLAB  : Fundamental domain aolution label (integer). The integer label
%         assigned by COCO to an trajectory segment during the continuation run
%         RUN.
%
% See also: ODE_ISOL2ISCRN, ODEISOL2COLL, ODE_COLL2COLL

%   % Copyright (C) James Hannam, Bernd Krauskopf, Hinke M. Osigna
%   $Id: ode_iscrn2iscrn.m 2020-11-26 15:57 NZDT$

grammar   = 'RUN [SOID] LAB [ORUN [OSOID] [OLAB]] [OPTS]';
args_spec = {
    'RUN', 'cell', '{str}',   'run',  {}, 'read', {}
   'SOID',     '',   'str',  'soid', oid, 'read', {}
    'LAB',     '',   'num',   'lab',  [], 'read', {}
   'ORUN', 'cell', '{str}',  'orun',  {}, 'read', {}
  'OSOID',     '',   'str', 'osoid',  {}, 'read', {}
   'OLAB',     '',   'num',  'olab',  [], 'read', {}
  };
opts_spec = {
%   '-corr_step', 'corr_step', false, 'toggle', {}
  };
[ args, opts ] = coco_parse( grammar, args_spec, opts_spec, varargin{:} );

[ sol, data ] = iscrn_read_solution( args.soid, args.run, args.lab );

%% this far - solution tobe constructed in read solution, such that a tangent
% vector can be passed to coll_add.

% Extra arguments for fundamental domain construction
tbid = coco_get_id( args.soid, 'iscrn' );
iscrn_data = coco_read_solution( tbid, args.run, args.lab, 'data' );
if ~(iscrn_data.fund) %&& (opts.corrStep)    % concatenate solutions
  cid1  = coco_get_id( tbid, 'coll' );
  data1 = coco_read_solution( cid1, args.run, args.lab, 'data' );
  maps1 = data1.coll_seg.maps;
  tbid2 = coco_get_id( args.osoid, 'iscrn' );
  cid2  = coco_get_id( tbid2, 'coll' );
  [ data2, chart2 ] = coco_read_solution( cid2, args.orun, args.olab,...
                       'data', 'chart' );
  seg2  = data2.coll_seg;
  maps2 = seg2.maps;
  mesh2 = seg2.mesh;

  tt      = chart2.x(maps2.T0_idx) + ...
              chart2.x(maps2.T_idx) * mesh2.tbp;
  tbp2    = tt(maps2.tbp_idx);
  xbp2    = transpose( reshape( chart2.x(maps2.xbp_idx), maps2.xbp_shp ) );
  xbp2    = xbp2(maps2.tbp_idx,:);

  sol.tbp = [ sol.tbp( 1:(end-1), : ); sol.tbp(end) + tbp2 ];
  sol.T   = sol.T + chart2.x(maps2.T_idx);
  sol.xbp = [ sol.xbp( 1:(end-1), : ); xbp2 ];
  % [ xbp, T0, T, p ]    % xbp including all tbp from mesh
  sol.t   = [ sol.t( maps1.xbp_idx( 1:(end-2) ) ); chart2.t(maps2.xbp_idx);...
              sol.t( ( maps1.xbp_idx(end) + 1 ) : end ) ]; % add fund.t that correspond to fund.xbp
  sol.u   = [ sol.u( maps1.xbp_idx( 1 : (end-2) ) ); chart2.x(maps2.xbp_idx);...
              sol.T0; sol.T; sol.u( ( maps1.xbp_idx(end) + 3 ) : end ) ];
  xbp_t   = reshape( chart2.t(maps2.xbp_idx), maps2.xbp_shp )';
  sol.xbp_t = [ sol.xbp_t(1:(end-1),:); xbp_t(maps2.tbp_idx,:) ];
end
% get tangent to x0 (tangent to isochron)
fid    = coco_get_id( tbid, 'x0' );
chartx = coco_read_solution( fid, args.run, args.lab, 'chart' );
close_data.x0_x0    = chartx.x;
close_data.x0_x0(1) = 0;
close_data.x0_t0    = chartx.t;
% We restart at branch point but stay on original solution manifold =>
% reset start direction to tangent vector of primary branch.
sol.t0     = sol.t;
sol.xbp_t0 = sol.xbp_t;
sol.T0_t0  = sol.T0_t;
sol.T_t0   = sol.T_t;
sol.p_t0   = sol.p_t;

if ~coco_exist('NullItMX', 'class_prop', prob, 'cont')
  % Compute improved tangent to new branch on same solution manifold to
  % allow change of parameters on restart at a branch-point.
  prob = coco_set(prob, 'cont', 'NullItMX', 1);
end

data.pr.oid = coco_get_id( oid, 'iscrn' );
tbid = coco_get_id( oid, 'iscrn' );
data = ode_init_data( prob, data, tbid, 'coll' );

prob = coll_add( prob, data, sol );

% prepare data for boundary conditions
close_data.xdim      = iscrn_data.xdim;
close_data.pnames    = iscrn_data.pnames;
close_data.s         = sol.s;
close_data.gamma_s   = sol.gamma_s;
close_data.tau_max   = sol.tau_max;
close_data.delta_max = [];
close_data.iscrn_par = iscrn_data.iscrn_par;
if iscrn_data.arc  % re-assemble arc arguments form previous run
  close_data.arc     = [ sol.l, iscrn_data.l_max, {iscrn_data.arc_par}, sol.x0 ];
else
  close_data.arc     = iscrn_data.arc;
end
close_data           = iscrn_init_data(prob, close_data, oid); % initialize 'iscrn' instance data structure
close_data.cids      = coco_get_id(tbid, 'coll');
prob = iscrn_close(prob, close_data);

prob = ode_add_tb_info(prob, oid, 'iscrn', 'seg', 'iscrn', iscrn_sol_info());
% prob = ode_add_tb_info(prob, oid, 'coll', 'seg', 'coll', coll_sol_info());

% if opts.corrStep
%   prob = coco_set(prob, 'cont', 'ItMX', [0, 1]);
% end
end
