function [ sol, data ] = iscrn_read_solution( oid, run, lab )
%ISCRN_READ_SOLUTION   Read 'ISCRN' solution and toolbox data from disk.
%
% Extract data structure associated with 'ISCRN' instance identifier and
% construct solution structure using trajectory segments from individual
% 'coll' instances.
%
% [SOL DATA] = ISCRN_READ_SOLUTION(OID, RUN, LAB)
%
% SOL  - Solution (struct).
% DATA - Toolbox data (struct).
% OID  - Object instance identifier (string).
% RUN  - Run identifier (string).
% LAB  - Solution label (integer).
%
% On output:
%
% SOL  : Solution structure.
% DATA : Toolbox data structure.
%
% COLL_READ_SOLUTION reconstructs the solution and toolbox data structures of a
% saved trajectory segment. More specifically, denote with
%
%   'seg'     :  a branch of trajectory segments and
%
% DATA will be a corresponding read-only instance of COCO_FUNC_DATA and
% always contain the fields of the ODE toolbox family.
%
% The solution structure SOL will have the fields
%
% SOL.tbp     : time discretization
% SOL.xbp     : sampled time history
% SOL.T       : interval duration
% SOL.p       : problem parameters
% SOL.s       : Fundamental domain, normalised.
% SOL.gamma_s : Base point of the fundamental domain.
% SOL.tau     : Value of tau at final point on isochron arc.
% SOL.tau_max : Norm of fundamental domain; scale of continuation variable tau.
%
% and additional fields encoding an initial solution point as required by
% COLL_ADD. If SOL results from the computation of the fundamental domain, the
% fields
%
% SOL.xbp_t     : Sampled tangent vector along bifucation digram at solution.
% SOL.T0_t      : Continuation tangent vector for T0.
% SOL.T_t       : Continuation tangent vector for T.
% SOL.p_t       : Continuation tangent vector for p.
% SOL.gamma_th  : Base point of Floquet vector.
% SOL.mu        : Stable(ised) Floqeut multiplier of (un)stable oscillation.
% SOL.w         : Floquet vector of relative phase.
% SOL.eta       : Rescaled distance along Floquet vector for base point of the
%                  fundamental domain.
% SOL.eta_max   : Scale of eta.
% SOL.delta     : Value of the rescaled continuation parameter delta.
% SOL.delta_max : Maximum orthogonal distance of start point on isochronal
%                  return map from Floquet vector; scale of delta and defines
%                  funndamental domain.
% SOL.s         : Fundamental domain, normalised.
% SOL.gamma_s   : Base point of the fundamental domain.
% SOL.tau_max   : Norm of fundamental domain; scale of continuation variable tau.
% SOL.NTST      : Number of mesh points in discretisation.
%
% will also be present. Additionally the fields
%
% ARC.l     : Array of arclength corresponding to points in COLS
% ARC.l_max : Maximum arclength at which to stop computation.
%
% are included when arclength is tracked. Depending on the types of the solution
% branch, the return value of SOL will have the following additional fields:
%
%   'seg.var' : The structure SOL.var will be initialized with restart data
%      for coll_add_var.
%
% See also: COLL_READ_SOLUTION, RPP_READ_SOLUTION, FLQT_READ_SOLUTION

%   % Copyright (C) James Hannam, Bernd Krauskopf, Hinke M. Osigna
%   $Id: iscrn_read_solution.m 2020-11-26 14:59 NZDT$

tbid = coco_get_id( oid, 'iscrn' );
cid  = coco_get_id( tbid, 'coll' );
try
  info = coco_read_tb_info( oid, run, lab, 'iscrn' );
%   if isempty( info )
%     info = coco_read_tb_info( oid, run, lab, 'coll' );
%   end
catch
  info = coco_read_tb_info( oid, run, lab, 'seg' );
end
format      = info.format;
branch_type = info.branch_type;

[ data, chart, uidx ] = coco_read_solution( cid, run, lab, 'data', ...
                          'chart', 'uidx' );

sol = struct( 'format', format, 'branch_type', branch_type, ...
  'pt_type', chart.pt_type, 'u', chart.x, 't', chart.t );

switch format

  case 'SOL.v1'
    [ sol, data ] = read_iscrn_v1( sol, data, chart, uidx, tbid, ...
      branch_type, run, lab );

  case 'iscrn.v1'
    [ sol, data ] = read_iscrn_v1( sol, data, chart, uidx, tbid, ...
      branch_type, run, lab );
end

end

function [ sol, data ] = read_iscrn_v1( sol, data, chart, uidx, tbid,...
                           branch_type, run, lab )  %#ok<INUSL>

seg  = data.coll_seg;
maps = seg.maps;
mesh = seg.mesh;

sol.T0  = sol.u(maps.T0_idx);
sol.T   = sol.u(maps.T_idx);
sol.tbp = sol.T0 + sol.T * mesh.tbp(maps.tbp_idx);
xbp     = reshape( sol.u(maps.xbp_idx), maps.xbp_shp )';
sol.xbp = xbp(maps.tbp_idx,:);
sol.p   = sol.u(maps.p_idx);

% tangent vector
xbp_t     = reshape(sol.t(maps.xbp_idx), maps.xbp_shp)';
sol.xbp_t = xbp_t(maps.tbp_idx,:);
sol.T0_t  = sol.t(maps.T0_idx);
sol.T_t   = sol.t(maps.T_idx);
sol.p_t   = sol.t(maps.p_idx);

% fundamental domain data - for reference purposes
iscrn_data = coco_read_solution( tbid, run, lab, 'data' );  % get data from iscrn restraint functions for later query
bd         = coco_bd_read( run );  % get bifurcation diagram for later query
if iscrn_data.fund    % last iteration was to find the fundamental domain
  s             = diff( sol.xbp([end,1],:) );    % s(0) - s(T)
  sol.tau_max   = norm( s );
  sol.s         = s / ( sol.tau_max );
  sol.gamma_s   = sol.xbp(end,:);
  sol.gamma_th  = iscrn_data.gamma_s;
  sol.w         = iscrn_data.s;
  sol.eta       = coco_bd_val( bd, lab, iscrn_data.iscrn_par{2} );
  sol.eta_max   = iscrn_data.tau_max;
  sol.mu        = iscrn_data.mu;
  sol.delta     = coco_bd_val( bd, lab, iscrn_data.fund_par{1} );
  sol.delta_max = iscrn_data.delta_max;
  if iscrn_data.psi_mode  % if running in psi_mode
    sol.psi        = iscrn_data.psi;
    sol.gamma_psi  = iscrn_data.gamma_psi;
    sol.w_psi      = iscrn_data.w_psi;
    sol.theta_diff = iscrn_data.dtheta;
  end
else
  % isochron information
  sol.s       = iscrn_data.s;
  sol.gamma_s = iscrn_data.gamma_s;
  sol.tau     = coco_bd_val( bd, lab, iscrn_data.iscrn_par{2} );
  sol.tau_max = iscrn_data.tau_max;
end

% isochron information
x0 = NaN( 1, numel( sol.xbp(1,:) ) );
for i = 1 : numel( sol.xbp(1,:) )
  x0(i) = coco_bd_val( bd, lab, iscrn_data.iscrn_par{i+2} );
end
sol.x0   = x0;
sol.NTST = data.coll.NTST;

% arc-length information
if iscrn_data.arc
  sol.l = coco_bd_val( bd, lab, iscrn_data.arc_par{1} );
end

end
