function iscrn = iscrn_read_isochron( arc_fmt, varargin )
%ISCRN_READ_ISOCHRON   Read 'iscrn' bifurcation diagram object from disk.
%
% Extract data structure associated with 'iscrn' instance identifier and
% extracts collumns of bifucation diagram object and isochron data.
%
% ISCRN    = ISCRN_READ_ISCORHON(RUN, VARARGIN, OPTS)
% VARARGIN = { [OID_FMT], COLS }
% OPTS = { '-trunc' TRUNC }
%
% ISCRN   - Solution (struct).
% RUN     - Run identifier (string).
% OID_FMT - Object instance identifier format (string); per sprintf formatting.
% COLLS   - Cell array of continuation parameter names (string).
% OPTS    - As described below.
% -trunc : indicates that extracted isochron data should be truncatedd at either
%         either the first point (initial) or last point (final) when
%         concatenating isochron arc. The string TRUNC = 'initial' or 'final',
%         as describedd above.
%
% An example of OID_FMT when using object instance identifiers such as 'T02' is
% 'T%02d'.
%
% On output:
%
% ISCRN : Solution structure with the fields
%
% ISCRN.iscrn     : Cell array of bifurcation data corresponding to COLLS.
% ISCRN.T         : Periodic of oscillation.
% ISCRN.p         : State parameters.
% ISCRN.tpo       : Time discretisation of oscillation.
% ISCRN.xpo       : Sampled time history of oscillation, roatatedd to raltive phase.
% ISCRN.tbp_s     : Time discretisation of fundamental solution.
% ISCRN.xbp_s     : Sampled time history of fundamental solution of relative phase.
% ISCRN.mu        : Stable(ised) Floqeut multiplier of (un)stable oscillation.
% ISCRN.w         : Floquet vector of relative phase.
% ISCRN.gamma_th  : Base point of Floquet vector.
% ISCRN.eta       : Rescaled distance along Floquet vector for base point of the
%                  fundamental domain.
% ISCRN.eta_max   : Scale of eta.
% ISCRN.delta     : Value of the rescaled continuation parameter delta.
% ISCRN.delta_max : Maximum orthogonal distance of start point on isochronal
%                  return map from Floquet vector; scale of delta and defines
%                  funndamental domain.
% ISCRN.s         : Fundamental domain, normalised.
% ISCRN.gamma_s   : Base point of the fundamental domain.
% ISCRN.tau       : Value of tau at final point on isochron arc.
% ISCRN.tau_max   : Norm of fundamental domain; scale of continuation variable tau.
% ISCRN.NTSTMN    : MInimum number of mesh points used in continuation.
% ISCRN.NTSTMX    : MAximum number of mesh point used in continuation
% ISCRN.TOL       : Tolerance used during continuation.
% ISCRN.stable    : Binary vlaue insdicating stability of oscillator.
% ISCRN.arc_idx   : Indicies of concatenation of isochron arcs.
%
% if OID_FMT is provided, otherwise only the fields iscrn and arc_idx are
% included. Additionally the fields
%
% ARC.l     : Array of arclength corresponding to points in COLS
% ARC.l_max : Maximum arclength at which to stop computation.
%
% are included when arclength is tracked.
%
% See also: ISCRN_READ_ISOCHRON, ISCRN_READ_FOLIATION

%   % Copyright (C) James Hannam, Bernd Krauskopf, Hinke M. Osigna
%   $Id: iscrn_read_isochron.m 2020-11-26 16:00 NZDT$

%% Parse arguments

grammar   = ' [OID_FMT] BD_COLS [OPTS]';
args_spec = {
  'OID_FMT',     '',   'str',    'oid_fmt', [], 'read', []
  'BD_COLS', 'cell', '{str}',    'bd_cols', {}, 'read', {}
  };
opts_spec = {
  '-trunc', 'trunc', 'none', 'read', []
  };
arg_stream   = coco_stream(varargin{:});

[args, opts] = coco_parse(grammar, args_spec, opts_spec, arg_stream);


% arc_fmt : dir/iscrn_nam/arc_nam%0k.0f+, dir/iscrn_nam/arc_nam+%0k.0f, dir/iscrn_nam/arc_nam%0k.0f
% oid_fmt : oid(k)
% k = 0 : fundamental arc

% consider truncatin of isochrons at concatenation points, but which side
% to truncate?

%% Read isochron

% find number of arcs by reading number of folders in directory that follow arc_fmt
i = find( arc_fmt == '/', 1, 'last' );  % find position of file separator
j = find( any( arc_fmt == ('+-')' ) , 1, 'last' );  % find position of side specifier in arc name
if isempty( j )  % if no side specifier, set j < i such that it triggers search by iscrn_nam only
  j = i - 1;
end
arc_nam = sprintf( arc_fmt, 0 );
num = ('0123456789')';
l = find( any( arc_nam == num ) );  % find indicies of arc specifiers
m = find( diff( l ) ~= 1, 1, 'last' ) + 1;  % index last change between numbers and characters
if isempty(m)  % if numbers appear only once in name
  m = 1;  % then use the index of the first number in string
end
% generate wildcards to count arcs in directory - see assumed naming convention in function help above
if j > l(m)  % if arc specifier is before side specifier
  files = dir( sprintf( '**/%s*%s*', arc_nam(1:(l(m)-1)), arc_fmt(j) ) );
  arcs  = length( { files.name } );
elseif j > i  % side specifier comes after file separator
  files = dir( sprintf( '**/%s%s*', arc_nam(1:(l(m)-1)), arc_fmt(j) ) );
  arcs  = length( { files.name } );
else  % side specifier is part of iscrn_nam
  files = dir( sprintf( '**/%s*', arc_nam(1:(l(m)-1)) ) );
  arcs = length( { files.name } );
end
% check that the zeroth file exists
koff = 0;  % offset to k in case that the zeroth file doesn't exist
if ~any( strcmp( arc_nam((i+1):end), {files.name} ) )
  koff = 1;
end

% read first arc - may be fundamental domain, keep special outputs if oid is given
if ~isempty( args.oid_fmt )
  oid     = sprintf( args.oid_fmt, 0 + koff );
  varargs = [ oid, varargin(2:end) ];
else
  varargs = varargin;
end
iscrn         = iscrn_read_arc( arc_nam, varargs{:} );
iscrn.arc_idx = NaN( arcs, 2 );  % store idicies of intial and final points of arc
iscrn.arc_idx = [ 1, length( iscrn.iscrn ) ];
for k = 1 : ( arcs - 1 )
  % read subsequent arcs
  arc_nam = sprintf( arc_fmt, k + koff );
  if ~isempty( args.oid_fmt )
    oid     = sprintf( args.oid_fmt, k + koff );
    varargs = [ oid, varargin(2:end) ];
  end
  ark = iscrn_read_arc( arc_nam, varargs{:} );
  % concatenate solutions
  switch opts.trunc  % optional truncation at concatenation points of isochron
    case 'initial'
      iscrn.iscrn          = vertcat( iscrn.iscrn, ark.iscrn(2:end,:) );
      if isfield( ark, 'l' )
        iscrn.l            = vertcat( iscrn.l, ark.l(2:end) );
      end
      iscrn.arc_idx(k+1,:) = [ 1, length( ark.iscrn ) - 1 ] + iscrn.arc_idx(end,2);
    case 'final'
      iscrn.iscrn          = vertcat( iscrn.iscrn(1:(end-1),:), ark.iscrn );
      if isfield( ark, 'l' )
        iscrn.l            = vertcat( iscrn.l, ark.l(1:(end-1)) );
      end
      iscrn.arc_idx(k+1,:) = [ 1, length( ark.iscrn ) ] + iscrn.arc_idx(end,2) - 1;
    otherwise
      iscrn.iscrn          = vertcat( iscrn.iscrn, ark.iscrn );
      if isfield( ark, 'l' )
        iscrn.l            = vertcat( iscrn.l, ark.l );
      end
      iscrn.arc_idx(k+1,:) = [ 1, length( ark.iscrn ) ] + iscrn.arc_idx(end,2);
  end
  if ~isempty( args.oid_fmt )
    iscrn.tau(k+1)       = ark.tau;
    iscrn.TOL(k+1)       = ark.TOL;
    iscrn.NTSTMN(k+1)    = ark.NTSTMN;
    iscrn.NTSTMX(k+1)    = ark.NTSTMX;
  end
end
% collapse settings that remain the same (these generally shouldn't be changed)
if ~isempty( args.oid_fmt ) && ~any( iscrn.TOL - iscrn.TOL(1) )
  iscrn.TOL = iscrn.TOL(1);
end

end
