function fol = iscrn_read_foliation( iscrn_fmt, varargin )
%ISCRN_READ_FOLIATION   Read 'iscrn' bifurcation diagram object from disk.
%
% Extract data structure associated with 'iscrn' instance identifier and
% extracts collumns of bifucation diagram object and isochron data for a
% foliation of isochrons.
%
% FOL    = ISCRN_READ_FOLIATION(RUN_FMT, VARARGIN)
% VARARGIN = { [OID_FMT], COLS, OPTS }
% OPTS = { '-trunc' TRUNC }
%
% FOL     - Solution (struct array).
% RUN_FMT - Run identifier format (string); per sprintf formatting.
% OID_FMT - Object instance identifier format (string); per sprintf formatting.
% COLLS   - Cell array of continuation parameter names (string).
% OPTS    - As described below.
% -trunc : indicates that extracted isochron data should be truncatedd at either
%         either the first point (initial) or last point (final) when
%         concatenating isochron arc. The string TRUNC = 'initial' or 'final',
%         as describedd above.
%
% An example of RUN_FMT for isochron data stored in directories such as
% 'iscrn001/arc02+' is 'iscrn%03d/arc%02d+'. Likewise, an example for OID_FMT
% when using object instance identifiers such as 'T02' is 'T%02d'.
%
% On output:
%
% ISCRN : Solution structure with the fields
%
% ISCRN.iscrn     : Cell array of bifurcation data corresponding to COLLS.
% ISCRN.T         : Periodic of oscillation.
% ISCRN.p         : State parameters.
% ISCRN.tpo       : Time discretisation of oscillation.
% ISCRN.xpo       : Sampled time history of oscillation, roatatedd to raltive phase.
% ISCRN.tbp_s     : Time discretisation of fundamental solution.
% ISCRN.xbp_s     : Sampled time history of fundamental solution of relative phase.
% ISCRN.mu        : Stable(ised) Floqeut multiplier of (un)stable oscillation.
% ISCRN.w         : Floquet vector of relative phase.
% ISCRN.gamma_th  : Base point of Floquet vector.
% ISCRN.eta       : Rescaled distance along Floquet vector for base point of the
%                  fundamental domain.
% ISCRN.eta_max   : Scale of eta.
% ISCRN.delta     : Value of the rescaled continuation parameter delta.
% ISCRN.delta_max : Maximum orthogonal distance of start point on isochronal
%                  return map from Floquet vector; scale of delta and defines
%                  funndamental domain.
% ISCRN.s         : Fundamental domain, normalised.
% ISCRN.gamma_s   : Base point of the fundamental domain.
% ISCRN.tau       : Value of tau at final point on isochron arc.
% ISCRN.tau_max   : Norm of fundamental domain; scale of continuation variable tau.
% ISCRN.NTSTMN    : MInimum number of mesh points used in continuation.
% ISCRN.NTSTMX    : MAximum number of mesh point used in continuation
% ISCRN.TOL       : Tolerance used during continuation.
% ISCRN.stable    : Binary vlaue insdicating stability of oscillator.
% ISCRN.arc_idx   : Indicies of concatenation of isochron arcs.
%
% if OID_FMT is provided, otherwise only the fields iscrn and arc_idx are
% included. Additionally the fields
%
% ARC.l     : Array of arclength corresponding to points in COLS
% ARC.l_max : Maximum arclength at which to stop computation.
%
% are included when arclength is tracked.
%
% See also: ISCRN_READ_ISOCHRON, ISCRN_READ_FOLIATION

%   % Copyright (C) James Hannam, Bernd Krauskopf, Hinke M. Osigna
%   $Id: iscrn_read_foliation.m 2020-11-26 16:00 NZDT$

%% Read isochron

% find number of isochrons by reading number of folders in directory that follow iscrn_fmt
i  = find( iscrn_fmt == '/', 1, 'last' );  % find position of file separator
ii = min( find( iscrn_fmt == '/', 2, 'last' ) );  % find position of second to last file separator
if ii == i  % if there is only one file separator
  ii = 0;  % then set ii to the first character
end
j  = find( any( iscrn_fmt == ('+-')' ) , 1, 'last' );  % find position of side specifier in iscrn name
if isempty( j )  % if no side specifier, set j > i such that it triggers search by iscrn_nam only
  j = i + 1;
end
iscrn_nam = sprintf( iscrn_fmt(1:(i-1)), 0 );
num = ('0123456789')';
l = find( any( iscrn_nam == num ) );  % find indicies of phase specifiers
m = find( diff( l ) ~= 1, 1, 'last' ) + 1;  % index last change between numbers and characters
if isempty(m)
  m = 1;
end
% generate wildcards to count arcs in directory - see assumed naming convention in function help above
if j > i  % if file separator is before side specifier
  files  = dir( sprintf( '**/%s*%', iscrn_nam(1:(l(m)-1)) ) );
  phases = length( { files.name } );
elseif j > l(m)  % phase specifier is before side specifier (and file separator)
  files  = dir( sprintf( '**/%s*%s', iscrn_nam(1:(l(m)-1)), iscrn_fmt(j) ) );
  phases = length( { files.name } );
else  % side specifier is before pahse specifier (and before file separator)
  files  = dir( sprintf( '**/%s%s*', iscrn_nam(1:(l(m)-1)), iscrn_fmt(j) ) );
  phases = length( { files.name } );
end
% check that the zeroth file exists
phioff = -1;  % offset to phi in case that the zeroth file doesn't exist
if ~any( strcmp( iscrn_nam((ii+1):end), {files.name} ) )
  phioff = 0;
end

% read first isochron to preallocate fol
iscrn_nam      = sprintf( iscrn_fmt(1:(i-1)), 1 + phioff );  % generate iscrn_nam
arc_nam        = sprintf( '%s%s', iscrn_nam, iscrn_fmt(i:end) );
iscrn          = iscrn_read_isochron( arc_nam, varargin{:} );
fields         = fieldnames( iscrn );
nulls          = cell( numel( fields ), 1 );
nulfields      = transpose( horzcat( fields, nulls ) );
fol(1:phases)  = struct( nulfields{:} );
fol(1)         = deal(iscrn) ;
% read isochrons in foliation
for phi = 2 : phases
  iscrn_nam = sprintf( iscrn_fmt(1:(i-1)), phi + phioff );  % generate iscrn_nam
  arc_nam   = sprintf( '%s%s', iscrn_nam, iscrn_fmt(i:end) );
  iscrn     = iscrn_read_isochron( arc_nam, varargin{:} );
  fol(phi)  = deal( iscrn );
end

end
