function arc = iscrn_read_arc( run, varargin )
%ISCRN_READ_ARC   Read 'iscrn' bifurcation diagram object from disk.
%
% Extract data structure associated with 'iscrn' instance identifier and
% extracts collumns of bifucation diagram object and isochron data.
%
% ARC = ISCRN_READ_ARC(RUN, VARARGIN)
% VARARGIN = { [OID], COLS }
%
% ARC   - Solution (struct).
% RUN  - Run identifier (string).
% OID   - Object instance identifier (string).
% COLLS - Cell array of continuation parameter names (string).
%
% On output:
%
% ARC : Solution structure with the fields
%
% ARC.iscrn   : Cell array of bifurcation data corresponding to COLLS.
% ARC.T       : Periodic of oscillation.
% ARC.p       : State parameters.
% ARC.s       : Fundamental domain, normalised.
% ARC.gamma_s : Base point of the fundamental domain.
% ARC.tau     : Value of tau at final point on isochron arc.
% ARC.tau_max : Norm of fundamental domain; scale of continuation variable tau.
% ARC.NTSTMN  : MInimum number of mesh points used in continuation.
% ARC.NTSTMX  : MAximum number of mesh point used in continuation
% ARC.TOL     : Tolerance used during continuation.
% ARC.stable  : Binary vlaue insdicating stability of oscillator.
%
% if OID is suppliedd, otherwise only iscrn is present. When extracting the
% fundamental arc of the isochron, the fields
%
% ISCRN.tpo       : Time discretisation of oscillation.
% ISCRN.xpo       : Sampled time history of oscillation, roatatedd to raltive phase.
% ISCRN.tbp_s     : Time discretisation of fundamental solution.
% ISCRN.xbp_s     : Sampled time history of fundamental solution of relative phase.
% ISCRN.mu        : Stable(ised) Floqeut multiplier of (un)stable oscillation.
% ISCRN.w         : Floquet vector of relative phase.
% ISCRN.gamma_th  : Base point of Floquet vector.
% ISCRN.eta       : Rescaled distance along Floquet vector for base point of the
%                  fundamental domain.
% ISCRN.eta_max   : Scale of eta.
% ISCRN.delta     : Value of the rescaled continuation parameter delta.
% ISCRN.delta_max : Maximum orthogonal distance of start point on isochronal
%                  return map from Floquet vector; scale of delta and defines
%                  funndamental domain.
%
% are also present. Additionally the fields
%
% ARC.l     : Array of arclength corresponding to points in COLS
% ARC.l_max : Maximum arclength at which to stop computation.
%
% are present when when arclength is tracked.
%
% See also: ISCRN_READ_ISOCHRON, ISCRN_READ_FOLIATION

%   % Copyright (C) James Hannam, Bernd Krauskopf, Hinke M. Osigna
%   $Id: iscrn_read_arc.m 2020-11-26 16:00 NZDT$

%% Parse arguments

grammar   = ' [OID] BD_COLS [OPTS]';
args_spec = {
      'OID',     '',   'str',     'oid', [], 'read', []
  'BD_COLS', 'cell', '{str}', 'bd_cols', {}, 'read', {}
  };
opts_spec = {
     '-ep', 'ep', 0, 'read', []
  };
arg_stream   = coco_stream(varargin{:});

[args, opts] = coco_parse(grammar, args_spec, opts_spec, arg_stream);

%% Read isochron arc

bd        = coco_bd_read( run );
arc.iscrn = transpose( coco_bd_col( bd, args.bd_cols ) );

%% Read isochron data from solution
if ~isempty( args.oid )
  lab        = max( coco_bd_labs( bd ) );  % last label, regardless of direction of continuation
  sol        = iscrn_read_solution( args.oid, run, lab );
  tbid       = coco_get_id( args.oid, 'iscrn' );
  iscrn_data = coco_read_solution( tbid, run, lab, 'data' );  % get data from iscrn restraint functions for later query
  % construct data
  arc.T = sol.T;
  arc.p     = sol.p;
  if isfield( sol, 'delta' )
    sol0       = iscrn_read_solution( args.oid, run, 1 );
    if ~opts.ep
      arc.tpo = sol0.tbp;
      arc.xpo = sol0.xbp;
      theta   = iscrn_read_phase( run, 0 );
    else
      arc.x = sol0.xbp(1,:);
      theta = iscrn_read_phase( run, 1 );
    end
    % fundamental domain only variables
    arc.tbp_s     = sol.tbp;
    arc.xbp_s     = sol.xbp;
    arc.theta = theta;
    arc.mu        = sol.mu;
    arc.w         = sol.w;
    arc.gamma_th  = sol.gamma_th;
    arc.eta       = sol.eta;
    arc.eta_max   = sol.eta_max;
    arc.delta     = sol.delta;
    arc.delta_max = sol.delta_max;
    arc.s       = sol.s;
    arc.gamma_s = sol.gamma_s;
    arc.tau     = [];  % placed here to maintain order ahead of tau_max
    arc.tau_max = sol.tau_max;
    % if running in psi_mode
    if isfield( sol, 'psi' )
      arc.psi        = sol.psi;
      arc.gamma_psi  = sol.gamma_psi;
      arc.w_psi      = sol.w_psi;
      arc.theta_diff = sol.theta_diff;
    end
  else
    % isochron information
    arc.s       = sol.s;
    arc.gamma_s = sol.gamma_s;
    arc.tau     = sol.tau;
    arc.tau_max = sol.tau_max;
  end
  % arc length only variables
  arc_par = iscrn_data.arc_par{1};
  if isfield(sol, arc_par)
    arc.l     = transpose(coco_bd_col(bd, arc_par));
    arc.l_max = iscrn_data.l_max;
  end
  % settings
  set_data   = coco_read_solution( iscrn_data.cids, run, lab, 'data' );
  arc.TOL    = set_data.ode.TOL;
  arc.NTSTMN = set_data.coll.NTSTMN;
  arc.NTSTMX = set_data.coll.NTSTMX;
  arc.stable = 1;  % forward-time isochron
  fhan       = char( set_data.fhan );
  if numel( fhan ) >=7 &&  strcmp( fhan(1:7), '@(u,p)-' )
    arc.stable = 0;  % compare function handle to see if time was reversed to stabilise the object, return 1 for stable, 0 for unstable.
  end
end

end

function theta = iscrn_read_phase( run, epflag )
% read phase from solution
theta = str2num(run);

end
