function data = iscrn_init_data(prob, src_data, oid, varargin) %#ok<INUSL>
%ISCRN_INIT_DATA   Initialize 'iscrn' instance data structure.
%
% DATA = ISCRN_INIT_DATA(PROB, SRC, OID, VARARGIN)
% VARARGIN = { [NAME]... }
%
% This function is part of the toolbox developers interface and should
% typically not be used directly.
%
% Construct 'iscrn' instance data structure.
%
% Input arguments:
%
% PROB     : Continuation problem structure.
% SRC_DATA : Source data structure.
% OID      : Object identifier of 'iscrn' toolbox instance.
% VARARGIN : List of toolbox field names to copy or initialize.
%
% On return, DATA contains the following fields:
%
% oid       : Object identifier, set to OID.
% xdim      : Dimension of the vector field.
% pnames    : List of parameter names.
% s         : Fundamental domain, not normalised.
% gamma_s   : Base point of the fundamental domian.
% tau_max   : Norma of the fundamental domain.
% iscrn_par : Internal parameter names for 'iscrn' toolbox.
% no_save   : List of field names to be excluded by coco_save_data.
%
% and any fields with names listed in VARARGIN.
%
% The fields xdim, pnames, s, and gamma_s are copied from the source data
% structure SRC_DATA and must be present. The iscrn_par field is set by the
% '-iscrn_par' option in ODE_ISOL2ISCRN, otherwise set per defaults in this
% function. Any fields with names passed in VARARGIN are either copied from
% SRC_DATA if a field with this name is present in SRC_DATA, or initialized to
% the empty structure. The field no_save is initialized to the empty set and
% collects names of fields to be omitted by the slot function COCO_SAVE_DATA.
% Additionally, when the field delta_max is present in SRC_DATA, the fields
%
% delta_max : Value of delta_max used for isochron computation.
% mu        : Floquet mulitplier.
% eta_max   : Approximation of scaling of tau such that tau = 1 when
%            delta = delta_max
%
% will also be present. This is set using the '-fund' option of ODE_ISOL2ISCRN.
% The '-arc' option of the constructor sets the arc field of SRC_DATA, including
% the fields
%
% arc   : Binary value indicating of arclength is tracked.
% l     : Arclength of the first point on arc of the isochron.
% l_max : Arclength along the isochron at which to stop computation.
% x0    : Coordinates of the begining of this isochron arc.
%
% in DATA. The constructed data structure DATA is a protected instance of
% COCO_FUNC_DATA.
%
% See also ODE_SETTINGS, COCO_SAVE_DATA, COCO_FUNC_DATA.

%   % Copyright (C) James Hannam, Bernd Krauskopf, Hinke M. Osigna
%   $Id: iscrn_init_data.m 2020-11-26 16:01 NZDT$

data = coco_func_data('protect');
data.oid = oid;

fields = varargin;
for i=1:numel(fields)
  field = fields{i};
  if isfield(src_data, field)
    data.(field) = src_data.(field);
  else
    data.(field) = struct();
  end
end

data.xdim    = src_data.xdim;
data.pnames  = src_data.pnames;
data.s       = src_data.s / norm( src_data.s, 2 );
data.gamma_s = src_data.gamma_s;
data.tau_max = src_data.tau_max;

% determine fundamental domain computation parameters
if ~isempty(src_data.delta_max)
  data.fund      = 1;  % run fundamental domain construction.
  data.delta_max = src_data.delta_max{1};
  data.mu        = src_data.delta_max{2};
  eta_max        = abs((data.mu) * sqrt(data.delta_max));  % scaling of eta_max
  data.tau_max   = eta_max;
  if numel(src_data.delta_max) > 2  % if custom names for fundamental domain variables are given
    data.fund_par  = src_data.delta_max{3};
  else
    delta_n = cell(1, (data.xdim) - 1);
      for i = 1:(max([1, (data.xdim) - 2]))
        delta_n{i} = sprintf('delta_n%1.0f', i);
        % overwrite numbering if three-dimensional
        if (data.xdim) == 3
          delta_n{1} = 'delta_n';
        end
      end
    fund_par       = [{ 'delta', 'delta_th' }, delta_n];
  end
  data.fund_par    = fund_par;
  % store bundle and phase point to find the fundamental domina by t = theta - psi + T return map
  if ~isempty(src_data.psi)
    data.w_psi     = src_data.psi{2};
    data.gamma_psi = src_data.psi{3};
    data.psi       = src_data.psi{1};
    data.dtheta    = src_data.psi{end};
    if numel(src_data.psi) > 4
      data.psi_par = src_data.psi{4};
    else
      data.psi_par = 'T_psi';
    end
    data.psid      = src_data.psid;
    data.psi_mode  = 1;  % flag for activating 'psi mode'
  else
    data.psi      = [];
    data.psi_mode = 0;  % do not activate 'psi mode'
  end
else
  data.fund = 0;  % do not run fundamental domain construction.
  data.psi      = [];
  data.psi_mode = 0;  % do not activate 'psi mode'
end
% define defualt variable names if not supplied in '-iscrn_par'
if isempty(src_data.iscrn_par)
  axis  = cell(1, data.xdim);
  for i = 1:(data.xdim)
     axis{i}     = sprintf('x0_%1.0f', i);
  end
  data.iscrn_par = [{'T', 'tau'}, axis];
else    % if variable names are supplied, use those instead
  data.iscrn_par = src_data.iscrn_par;
end

% determine tracking of isochron arc length
if ~isempty(src_data.arc)
  data.arc   = 1;  % track arc length
  data.l     = src_data.arc{1};
  data.l_max = src_data.arc{2};
  data.x0    = src_data.arc{end};
  if numel(src_data.arc) == 3  % arc{end} = x0 is added in iscrn_init_args
    data.arc_par = {'l', 'ARC'};  % default arclegnth parameters
  else
    data.arc_par = src_data.arc{3};  % custom arclength parameters
  end
else
  data.arc = 0;  % do not track isochron arclength
end
% add tangent to isochron
if isfield(src_data, 'x0_t0')
  data.x0_x0 = src_data.x0_x0;
  data.x0_t0 = src_data.x0_t0;
end

data.no_save = {};

end
