function [sol, data] = flqt_read_solution(oid, run, lab)
%FLQT_READ_SOLUTION   Read 'flqt' solution and toolbox data from disk.
%
% Extract data structure associated with 'flqt' instance identifier and
% construct solution structure using trajectory segments from individual
% 'coll' instances.
%
% [SOL DATA] = FLQT_READ_SOLUTION(OID, RUN, LAB)
%
% SOL  - Solution (struct).
% DATA - Toolbox data (struct).
% OID  - Object instance identifier (string).
% RUN  - Run identifier (string).
% LAB  - Solution label (integer).
%
% On output:
%
% SOL  : Solution structure.
% DATA : Toolbox data structure.
%
% COLL_READ_SOLUTION reconstructs the solution and toolbox data structures of a
% saved trajectory segment. More specifically, denote with
%
%   'seg'     :  a branch of trajectory segments and
%   'seg.VAR' :  a branch of trajectory segments with variational problem.
%
% DATA will be a corresponding read-only instance of COCO_FUNC_DATA and
% always contain the fields of the ODE toolbox family.
%
% The solution structure SOL will have the fields
%
% SOL.tbp : time discretization
% SOL.xbp : sampled time history
% SOL.T   : interval duration
% SOL.p   : problem parameters
% SOL.mu  : Floquet multiplier
% SOL.w   : Floquet vector
%
% and additional fields encoding an initial solution point as required by
% COLL_ADD. If SOL results from rotation of the Floquet vector, the fields
%
% SOL.theta : relative phase of the solution
% SOL.wbp   : Floquet bundle sampled time history
% gamma_th  : phase point on periodic orbit
%
% will also be present. Depending on the types of the solution branch, the
% return value of SOL will have the following additional fields:
%
%   'seg.var' : The structure SOL.var will be initialized with restart data
%      for coll_add_var.
%
% See also: COLL_READ_SOLUTION, RPP_READ_SOLUTION

%   % Copyright (C) James Hannam, Bernd Krauskopf, Hinke M. Osigna
%   $Id: flqt_read_solution.m 2020-11-26 16:04 NZDT$

tbid = coco_get_id(oid, 'flqt');
data = coco_read_solution(tbid, run, lab);

% determine floquet bundle and multiplier
if data.nsegs == 1
  colloid   = coco_get_id(tbid, sprintf('seg%d', 1)); % 'segment' identifier
  segoid    = coco_get_id(colloid, 'coll'); % 'coll' object instance identifier
  varsegoid = coco_get_id(segoid,'var'); % object instance identifier
  sol       = coll_read_solution(colloid, run, lab);
  chart     = coco_read_solution(varsegoid, run, lab, 'chart');
  segdata   = coco_read_solution(segoid, run, lab, 'data');
  M         = chart.x(segdata.coll_var.v1_idx);    % scale out the perturbation length
  [w,mu]    = eig(M, 'vector');    % determine eigenvalue
  [mu,ind]  = min(mu);
  w         = w(:,ind);
  fhan = char(segdata.fhan);
  if (length(fhan) > 7) && strcmp(fhan(1:7), '@(u,p)-')  % correct eigenvalue for stabalised system
    sol.eigenvalues = 1 ./ mu;
  end
else
  % construct variational solution
  wbp   = cell(2,1);
  tbid2 = coco_get_id(tbid, 'rpp');    % keep track of cid through isol2rpp
  for i = [1,2]
    segoid    = coco_get_id(tbid2, sprintf('seg%d.coll', i));    % 'coll' object instance identifier
    varsegoid = coco_get_id(segoid,'var'); % object instance identifier
    chart     = coco_read_solution(varsegoid, run, lab, 'chart');
    segdata   = coco_read_solution(segoid, run, lab, 'data');
    v_idx     = segdata.coll_var.v_idx;    % indicies for variational solution set
    xbp_shp   = segdata.coll_seg.maps.xbp_shp;
    tbp_idx   = segdata.coll_seg.maps.tbp_idx;    % index of solution set that belongs to solution trajecotry
    w         = reshape(chart.x(v_idx), xbp_shp)';    % reshape solution set to dim-d array
    wbp{i}    = w(tbp_idx,:);    % only keep entries that correspond to trajecotry segment
  end
  % write extra data to solution
  bd    = coco_bd_read(run);
  mu    = coco_bd_val(bd, lab, data.flqt_par{2}{end});
  theta = coco_bd_val(bd, lab, data.flqt_par{1}{1});
  sol   = rpp_read_solution( tbid, run, lab );
  cdata = coco_read_solution( segoid, run, lab );
  rppid = coco_get_id(tbid, 'rpp');  % get oid for rpp part of solution
  rdata = coco_read_solution(rppid, run, lab);  % get data to determine data.zpp flag
  tol   = cdata.ode.TOL; % concatenate variational equation solution depending on data.mode
  switch data.mode
      case 1  % mode 1 - roate linear bundle of periodic orbit through
        % forward-time phase theta in a stabilised (backward-time) system
        if trivial_theta( theta, tol )  % truncate solution of 'constant' part when seg{i}.T=0
          if rdata.zpp
            sol.wbp = wbp{1};
          else
            sol.wbp = wbp{2};
          end
        else
          sol.wbp  = [wbp{1}; wbp{2}(2:end,:)];
        end
      case 2  % mode 2 - rotate linear bundle of periodic orbit through
        % forward-time phase given by theta
        if trivial_theta( theta, tol )  % truncate solution of 'constant' part when seg{i}.T=0
          wbp{1} = [];
        end
        sol.wbp = [wbp{2}; wbp{1}(2:end,:)];
  end
  w            = sol.wbp(1,:)';
  sol.theta    = theta;
  sol.gamma_th = sol.xbp(end,:);
end

% add to solution structure
sol.mu = mu;
sol.w  = w';

end

function flag = trivial_theta( theta, tol )
%TRIVIAL_THETA    Determine if phase on [0,1) is near zero.
% case is taken to account for phases near integer values that correspond
% to near zero phase. In these cases, the solution sol.xbp will likely
% contain a non-monotonous part corresponding to the segment with near zero
% period.

flag = false;  % set flag
if any( abs( [ theta; 1 - mod( theta, 1 ) ] ) <= tol )
  flag = true;
end

end
