function data = flqt_init_data(prob, src_data, oid, varargin) %#ok<INUSL>
%FLQT_INIT_DATA   Initialize 'flqt' instance data structure.
%
% DATA = FLQT_INIT_DATA(PROB, SRC, OID, VARARGIN)
% VARARGIN = { [NAME]... }
%
% This function is part of the toolbox developers interface and should
% typically not be used directly.
%
% Construct 'flqt' instance data structure and initialize with phases of
% phase points to be located
%
% Input arguments:
%
% PROB     : Continuation problem structure.
% SRC_DATA : Source data structure.
% OID      : Object identifier of 'flqt' toolbox instance.
% VARARGIN : List of toolbox field names to copy or initialize.
%
% On return, DATA contains the following fields:
%
% oid       : Object identifier, set to OID.
% xdim      : Dimension of vector field.
% pnames    : List of parameter names.
% theta     : List of phases for phase points to be located.
% flqt_par  : Internal Parameter names for 'flqt' toolbox.
% no_save   : List of field names to be excluded by coco_save_data.
%
% and any fields with names listed in VARARGIN.
%
% The fields xdim, pnames, and theta are copied from the source data structure
% SRC_DATA and must be present. The flqt_par field is set by the '-flqt_par'
% option in ODE_ISOL2FLQT, otherwise set per defaults in this function. Any
% fields with names passed in VARARGIN are either copied from SRC_DATA if a
% field with this name is present in SRC_DATA, or initialized to the empty
% structure. The field no_save is initialized to the empty set and collects
% names of fields to be omitted by the slot function COCO_SAVE_DATA. The
% constructed data structure DATA is a protected instance of COCO_FUNC_DATA.
%
% See also ODE_SETTINGS, COCO_SAVE_DATA, COCO_FUNC_DATA.

%   % Copyright (C) James Hannam, Bernd Krauskopf, Hinke M. Osigna
%   $Id: flqt_init_data.m 2020-11-26 14:55 NZDT$

data = coco_func_data('protect');
data.oid = oid;

fields = varargin;
for i=1:numel(fields)
  field = fields{i};
  if isfield(src_data, field)
    data.(field) = src_data.(field);
  else
    data.(field) = struct();
  end
end


data.xdim    = src_data.xdim;
data.pnames  = src_data.pnames;
data.mode    = src_data.mode;
data.nsegs   = src_data.nsegs;

if data.nsegs==2
  data.mu     = src_data.mu;
  data.w0     = src_data.w0;
end

% define defualt variable names if not supplied in '-flqt_par'
if isempty(src_data.flqt_par)
  axis       = cell(1,data.xdim);
  flqtnames  = axis;
  for i = 1:(data.xdim)
     axis{i}      = sprintf('%s%s', 'gama_', num2str(i));
     flqtnames{i} = sprintf('%s%s',  'wth_', num2str(i));
  end
  rpp_par       = [ {'theta', 'phi0', 'phith', 'THETA'}, axis ];
  data.flqt_par = { rpp_par; [ flqtnames, {'mu'} ] };
else    % if variable names are supplied, use those instead
  data.flqt_par = src_data.flqt_par;
end

data.no_save = {};

end
