function sol = ep_bundle(DFDX, x0, p, epsilon, theta, stab)
%%EQ_LIN_BUNDLE generates the linear bundle associated with an equilibrium
%%x0 at distance eps for angles theta around x0. Angle theta is defined
%%on (0,1], such that theta = 0 defines the maxima on the ellipse
%%associated with the bundle.

%   % Copyright (C) James Hannam, Bernd Krauskopf, Hinke M. Osigna
%   $Id: ep_bundle.m 2020-11-26 14:50 NZDT$

% allocate stability check
if ~exist('stab', 'var')
  stab = [];
end
if size(theta,1) > 1
  theta = transpose(theta);
end
% if unstable, stabalise and find corresponding forward-time phase
if strcmp(stab, '-u')
  DFDX  = @(u,p) -DFDX(u,p);
  theta = 1 - theta;
end

% Eigenvalues
J = DFDX(x0, p);
[v, l] = eig(J, 'vector');

c1 = 2 * abs( epsilon * v(:,1) );
c2 = angle( epsilon * v(:,1) );

% maximum of ellipse in the first co-ordanate
t_star = (2*pi-c2(1)) / ( 2*pi );
% if cos( 2*pi * t_star + c2(1) ) < 0
%   t_star = t_star + pi / ( 2*pi );
% end

w = c1 .* cos(2*pi * ( theta + t_star ) + c2);
w = transpose(w ./ sqrt(sum( w .* w, 1 )));    % normalise

% write solution
n = numel(theta);
b = abs(imag( l(1) ));
T = 2*pi / b;

sol.tbp     = [0; T];
sol.xbp     = [x0, x0]';
sol.p       = p;
sol.theta   = 1 - theta;
sol.w       = w;
sol.lambda  = l(1);
sol.T       = T;
sol.mu      = exp(-real(l(1)) * T);    % mu = exp( -alpha * T ) : required to rescale eta in fundamental domain computations
sol.epsilon = epsilon;
sol.eigenvalues  = l;
sol.eigenvectors = v;
if strcmp(stab, '-u')  % correct eigenvalue for stabalised system
  sol.eigenvalues = -l;
end

% if multiple solutions, contruct structure array of solutions
if n > 1
  sol.w     = [];  % don't copy large arrays to each instance of sol
  sol.theta = [];
  sol(1:n)  = deal(sol);
  for phi = 1:n
    sol(phi).w     = w(phi,:);
    sol(phi).theta = 1 - theta(phi);
  end
end

end
