%% Rotation of periodic orbits from SM1 in Isochron foliations and global
% bifurcations.
%
% Computation of an intial approximation of a periodic orbit in the dynamical
% system with vector field given in Eq. (10) in Isochron foliations and global
% bifurcations, following the instructions in SM1. Here, we use ODE45 and the
% CoCo-compatatable encoding of the vetor field to integrate an initial
% approximation of the periodic orbit, and then rotate that approximation using
% the 'rpp' toolbox such that its start-point lies at the maximum in the first
% coordinate of the vedtor field.

%% Integrating the periodic orbit
%
% The commands here specifically compute the stable periodic obrit shown in
% Fig. 2 in Isochron foliations and global bifurcations from region D. Hence,
% the variable 'p0' stores the system parameter values for this periodic orbit.
% The vector field given in Eq. (10) is encoded as the function TUBE(X,P) in a
% CoCo-compatable format. The first call to ODE45 integrates from a suitable
% intial condition for an integration time long enough that the resulting
% trajectory will converge to the stable periodic orbit. The next call to ODE45
% integrates from the final point of the trajectory from the previous
% integration for a time 10.5, which by visual inspection of that trajectory is
% near to the period of the periodic orbit. The resulting aprroximation of the
% periodic orbit is stored in t0 and x0.

p0       = [-0.66; 0.5; 2.5; 2.5; 1.0];
[~, x0]  = ode45(@(t, x) tube(x, p0), [0, 100 * pi], [0; 0.25]);
[t0, x0] = ode45(@(t, x) tube(x, p0), [0, 10.5], x0(end,:)');

%% Applying zero-phase convention
%
% The commands below rotate the approximation of the stable periodic orbit above
% such that it satisfies the zero-phase convention for isochrons. For this
% purpose alone, the constructor ODE_ISOL2RPP automaically calls the coco entry-
% point function, and the label number of the solution that satisfies the zero-
% phase convention is returned in the variable ind. Hence, the apropriatly
% rotated periodic orbit is extracted in the follwoing call to RPP_READ_SOLUTION.

p    = {'a', 'b', 'c', 'd', 'mu_s'};    % paramter names
stab = 's';
prob = coco_prob();
prob = orbit_settings(prob);

[~, PO_MAX, ind] = ode_isol2rpp(prob, 'po', @tube, @tube_DFDX, @tube_DFDP,...
  t0, x0, p, p0, '-zpp', 'zero_phase', stab);

sol = rpp_read_solution( 'po', 'zero_phase', ind);
t0  = sol.tbp;
x0  = sol.xbp;

%% Computation of Floquet vectors of a periodic orbit from SM1 in Isochron
% foliations and global bifurcations.
%
% Computation of Floquet vectors given an approximation of a periodic orbit in
% the dynamical system with vector field given in Eq. (10) in Isochron
% foliations and global bifurcations, following the instructions in SM1. Here,
% we use the 'flqt' toolbox to compute the Floquet vector for the start point on
% the periodic orbit computed in RPP/DEMO.M, which is run at the begining of
% this demo. This process also computes the Floquet multipler of the periodic
% orbit. We then rotate the periodic orbit and its Floquet vector, saving
% solutions for desired phases.

%% Computing Floqeut multiplier and vector
%
% The following commands compute the Floquet multipler and Floquet vector  for
% the stable periodic orbit supplied in the variables 't0', 'x0', and 'p0'.
% Specifically, we use the result produced from EXAMPLES/RPP/DEMO.M. The
% variable 'p0' stores the system parameter values for this periodic orbit. The
% vector field given in Eq. (10) is encoded as the function TUBE(X,P) in a
% CoCo-compatable format. The final line extracts the solution to 'sol0', a
% structure containing the periodic orbit ('tbp', 'xbp', and 'p') and the
% Floquet multiplier and Floquet vector ('mu', and 'w') for the start point on
% the periodic orbit.

prob  = coco_prob();
prob  = orbit_settings(prob);
prob  = ode_isol2flqt(prob, 'po', @tube, @tube_DFDX, @tube_DFDP,...
  t0, x0, p, p0);
FLQT0 = coco(prob, 'floquet0', [], 0);
sol0  = flqt_read_solution('po', 'floquet0', 1);

%% Rotating the Floquet vector
%
% Using 'sol0' from above as the intial solution, the following commands rotate
% the periodic orbit and its Floquet vector through phase in [0,1]. We set an
% array 'theta' of phases at which to save solutions for later use; we remove
% phase 1 from this array as it is a repeat of phase 0. All arguments passed to
% ODEISOL2FLQT before '-flqt0' are mandatory. This final optional argument sets
% the constructor for the rotation of the Floquet vector, and must be followed
% by a celly array containing the Floquet multipler anf Floqeut vector.

theta = 0 : (1 / 20) : 1;
theta(end) = [];
prob  = coco_prob();
prob  = orbit_settings(prob);
prob  = ode_isol2flqt(prob, 'po', @tube, @tube_DFDX, @tube_DFDP,...
  sol0.tbp, sol0.xbp, p, sol0.p, theta, '-flqt0', {sol0.mu; sol0.w});
FLQT  = coco(prob, 'floquet', [], 1, {'theta', 'mu', 'wth_1', 'wth_2'},...
         {[-1e-4, max(theta) + 1e-4], [], [], []});

%% Extracting a bundle of Floquet vectors
%
% The following commands extract solutions from the above continuation run into
% a cell array of structures 'sol', one cell per phase of solution.

THTlabs = coco_bd_labs(FLQT, 'THETA');
Nphase  = numel(THTlabs);
sol     = cell(Nphase, 1);
for phi = 1 : Nphase
  sol{phi} = flqt_read_solution('po', 'floquet', THTlabs(phi));
end

%% Computation of arcs of the zero-phase isochron of  a periodic orbit from SM1
% in Isochron foliations and global bifurcations.
%
% Computation of arcs of the zero-phase isochron of a periodic orbit in the
% dynamical system with vector field given in Eq. (10) in Isochron foliations
% and global bifurcations, following the instructions in SM1. Here, we use the
% 'iscrn' toolbox to compute the fundamental domain and first two arcs of the
% zero-phase isochron of the stable periodic orbit computed in RPP/DEMO.M, given
% the Floquet vector computed in FLQT/DEMO.M, which is run at the begining of
% this demo. This process may be extended to compute further arcs of the
% isochron, or isochrons of other phases. Alternatively, the script supplied in
% ISOCHRON_FOLIATION/ISOCHRON_FOLIATION.M may be used for a more automatic
% application.

%% Compute fundamental domain
%
% The following commands compute fundamental domain that will be used to compute
% the zero-phase isochron. We use the solution 'sol0' produced by FLQT/DEMO.M,
% the fields oif which contain the stable periodic orbit ('tbp', 'xbp', and 'p')
% Floquet multiplier and Floquet vector ('mu', and 'w') for the start point on
% the periodic orbit. The vector field given in Eq. (10) is encoded as the
% function TUBE(X,P) in a CoCo-compatable format. The variable 'deltamax' sets
% the error in the reutrn map that defines the zero-phase isochron in terms of
% orthogonal distance from the Floquet vector that the start point of the map is
% allowed to deviate. The function ISCRN_SETTINGS(PROB) is used to set
% the continuation settings, and the following line sets continuation to occur
% only in the 'forward' direction with respect to the primary continuation
% parameter. The arguments preceeding '-fund' are mandatory for ODE_ISOL2ISCRN,
% and this optional arguments insctructs the constructor to prepare computation
% of the fundamental domain of the isochron. The option '-fund' must be followed
% by a cell array containing 'deltamax', as describned above, and the floquet
% multiplier of the periodic orbit. The final optional argument '-arc' indicates
% the the arclength along the computed isochron should be tracked, and is
% followed by a cell array containing the minimum and maximum descired
% arclengths. The call to the CoCo entry-point function, and the first string
% argument instructs the directory in which data should be stored. The
% computational bounds of each active continuation paramter are given in the
% last argument, corresponding to the variables stated in the previous cell
% array fo strings. The variables 'tau' and 'delta' are both rescaled; we wish
% to stop continuation when 'delta' reaches 1 due to this rescaling.

deltamax = 5e-4;
prob = coco_prob();
prob = iscrn_settings(prob);
prob = coco_set(prob, 'cont', 'ItMX', [0, 100]);
prob = ode_isol2iscrn(prob, 'T0', @tube, @tube_DFDX, @tube_DFDP,...
  sol0.tbp, sol0.xbp, p, sol0.p, sol0.w, [],...
  '-fund', {deltamax; sol0.mu}, '-arc', {0, 20});
ISO{1} = coco(prob, 'Gs/arc00+', [], 1,...
  {'tau', 'delta', 'x0_1', 'x0_2', 'l'},...
  {[], [-1, 1], [], [], []});

%% Extracting the fundamental domain
%
% The following commands extract the short section of isochron computed while
% finding the fundamental domain of the isochron, and plot this short sgement.
% Also plotted, the change in 'delta' with respect to 'tau' during continuation.
% This data is extracted from the output of the coco entry-point function,
% using post-processing utilities provided by CoCo.

tau   = coco_bd_col(ISO{1}, 'tau');
delta = coco_bd_col(ISO{1}, 'delta');
x0    = coco_bd_col(ISO{1}, {'x0_1', 'x0_2'});
figure(1)
plot(tau, delta)
xlabel('tau')
ylabel('delta', 'rotation', 0)
figure(2)
plot(x0(1,:), x0(2,:))
xlabel('x')
ylabel('y', 'rotation', 0)

%% Compute the first arcs
%
% We use the fundamental domain computed above to construct the return map that
% defines the follwoing arcs of the zero-phase isochron. We set the number of
% mesh points for this computation to 50, and extract the number of mesh points
% used in the fundamental solution to 's_NTST'. For the first arc, the starting
% solution is the same as the solution from the fundamental domain, so 'NTST'
% setting of the 'coll' toolbox is set as such. The 'NTSTMX' setting sets the
% maximum number of mesh points, and is set as 'ntst'. The call to the
% constructor ODE_ISCRN2ISCRN extracts the solution above from the directory and
% object identifer supplied. The following call the to coco entry-point function
% intiates continuation, storing data in the 'Gs/arc01+' directory. The '-arc'
% option used when computing the fundamental domain is passed along by
% ODE_ISCRN2ISCRN, and 'tau' is scaled such that it varies from 0 to 1 as the
% start point of the isochronal return map moves along the fundamental domain.
%
% The second arc is computed using the terminal solution fo the first arc. The
% constructor ODE_ISCRN2ISCRN concatenates this terminal solution with the
% solution the defines the fundamental domian. This constructs the isochronal
% return map for twice the period of the stable periodic orbit to ehich the
% isochorn belongs. Hence, the number of mesh points for this initial solution
% is set as the addition of 'k_NTST' and 's_NTST', and 'NTSTMX' is set as twice
% 'ntst'. The first three arguments of ODE_ISCRN2ISCRN following prob reference
% the terminal solution fo the first arc, and the next three arguments reference
% the terminal solution of the fundamental arc. The follwoing call to the CoCo
% ebtry-point function intiates continuation.

% First arc
ntst   = 50;
s_NTST = coco_bd_val(ISO{1}, 2, 'T0.iscrn.NTST');
prob   = coco_prob();
prob   = iscrn_settings(prob);
prob   = coco_set(prob, 'coll', 'NTST', s_NTST, 'NTSTMX', ntst);
prob   = ode_iscrn2iscrn(prob, 'T1', 'Gs/arc00+', 'T0', 2,...
  'Gs/arc00+', 'T0', 2);
ISO{2} = coco(prob, 'Gs/arc01+', [], 1, {'tau', 'x0_1', 'x0_2', 'l'},...
  {[-0.1, 1], [-1,1], [-1,1], []});

% Second arc
k_NTST = coco_bd_val(ISO{2}, 2, 'T1.iscrn.NTST');
prob   = coco_prob();
prob   = iscrn_settings(prob);
prob   = coco_set(prob, 'coll', 'NTST', k_NTST + s_NTST, 'NTSTMX', 2 * ntst);
prob   = ode_iscrn2iscrn(prob, 'T2', 'Gs/arc01+', 'T1', 2, 'Gs/arc00+', 'T0', 2);
ISO{3} = coco(prob, 'Gs/arc02+', [], 1, {'tau', 'x0_1', 'x0_2', 'l'},...
  {[-0.1, 1], [-1,1], [-1,1], []});

%% Visualising isochron arcs
%
% The post-processing functions ISCRN_READ_SOLUTION, ISCRN_READ_ARC, and
% ISCRN_READ_ISOCHRON allow for the extraction of data produced by the 'iscrn'
% toolbox constructors. The utility ISCRN_READ_SOLUTION extracts the solution
% to the isochron boundary value problem given a run and label number. The
% ISCRN_READ_ARC utility instead extracts the isochron arc, more specifically
% the continuation variables requested in the cell array taken as its final
% argument. Below we extract the terminal solution of the fundamental arc, and
% the first arc of the zero-phase isochron, and then plot that arc in figure 1.
%
% The utility ISCRN_READ_ISOCHRON reads the arcs of the isochron stored in
% a directory structure formatted accrding to the string provided as its
% first argument. This string format is the same as used by MATLAB's SPRINTF.
% The second string argument is optional; its inclusion adds more data to the
% output strucrue 'iscrn'. Following the cell array of continuation vairbales to
% extract, the option '-trunc' truncates this data to avoid repetiion where the
% arcs of the isochron are joined. The follwoing commands plot the zero-phase
% isochron computed above, as well as plotting the arclength of the isochron
% against its  x-coordiante.

% Isochron arc and solution

sol = iscrn_read_solution('T0', 'Gs/arc00+', 2);
arc = iscrn_read_arc('Gs/arc01+', 'T1', {'x0_1', 'x0_2'});

figure(1)
plot(arc.iscrn(:,1), arc.iscrn(:,2))
xlabel('x')
ylabel('y', 'rotation', 0)

% Isochron

iscrn = iscrn_read_isochron('Gs/arc%02d+', 'T%d', {'x0_1', 'x0_2'},...
  '-trunc', 'initial');

figure(2)
plot(iscrn.iscrn(:,1), iscrn.iscrn(:,2), 'b')
hold on
plot(iscrn.xpo(:,1), iscrn.xpo(:,2), 'k')
xlabel('x')
ylabel('y', 'rotation', 0)

figure(3)
plot(iscrn.iscrn(:,1), iscrn.l, 'b')
xlabel('x')
ylabel('l', 'rotation', 0)
