%% Computation of arcs of the zero-phase isochron of an unstable equilibrium
% from SM1 in Isochron foliations and global bifurcations.
%
% Computation of arcs of the zero-phase isochron of an unstable equilibrium in
% the dynamical system with vector field given in Eq. (10) in Isochron foliations
% and global bifurcations, following the instructions in SM1. Here, we use the
% 'iscrn' toolbox to compute the fundamental domain and first two arcs of the
% zero-phase isochron of the unstable equilibrium.

% Add the parent directory to MATLAB's search path. The files iscrn_settings.m,
% tube.m, tube_DFDX.m, and tube_DFDP.m used in this and further demos are stored
% in the parent directory.
addpath('../')

%% Compute linear approximation
%
% The following commands compute fundamental domain that will be used to compute
% the zero-phase isochron. First, we compute the linear approximation of the
% isochron using EP_BUNDLE, which takes as arguments the Jacobian of the vector
% field, the equilibrium 'x0', the system parameter array 'p0', the radius of
% radius of the blow up, and the phase of the linear approximation desired.
%
%  Following this, we construct an cell array of structures, one cell entry for
% each phase of linear approximation desired.

% linear approximation for equilibria

x0   = [0.3253; -0.2414];
p0   = [-0.66; 0.5; 2.5; 2.5; 1.0];
sol0 = ep_bundle(@tube_DFDX, x0, p0, 1e-4, 0);

x0      = [0.3253; -0.2414];
p0      = [-0.66; 0.5; 2.5; 2.5; 1.0];
theta   = 0 : (1/20) : 0.99;
Nphase  = numel(theta);
sol     = cell(Nphase, 1);
for phi = 1 : Nphase
  sol{phi} = ep_bundle(@tube_DFDX, x0, p0, 1e-4, theta(phi));
end

%% Compute the fundamental domain
%
% We use the solution 'sol0' produced above the fields of which contain the
% unstable equilibrium ('tbp', 'xbp', and 'p'), the eigenvalue and linear
% approximation ('mu', and 'w') for the zero-phase isochrons. The vector field
% given in Eq. (10) is encoded as the function TUBE(X,P) in a CoCo-compatable
% format. The variable 'deltamax' sets the error in the reutrn map that defines
% the zero-phase isochron in terms of orthogonal distance from the Floquet
% vector that the start point of the map is allowed to deviate. The function
% ISCRN_SETTINGS(PROB) is used to set the continuation settings, and the
% following line sets continuation to occur only in the 'forward' direction with
% respect to the primary continuation parameter. The arguments preceeding '-fund'
% are mandatory for ODE_ISOL2ISCRN, and this optional arguments insctructs the
% constructor to prepare computation of the fundamental domain of the isochron.
% The option '-fund' must be followed by a cell array containing 'deltamax', as
% described above, and the unstable eigenvalue of the equilibrium. The final
% optional argument '-arc' indicates the the arclength along the computed
% isochron should be tracked, and is followed by a cell array containing the
% minimum and maximum descired arclengths. The call to the CoCo entry-point
% function, and the first string argument instructs the directory in which data
% should be stored. The computational bounds of each active continuation
% parameter are given in the last argument, corresponding to the variables
% stated in the previous cell array of strings. The variables 'tau' and 'delta'
%are both rescaled; we wish to stop continuation when 'delta' reaches 1 due to
% this rescaling.

% fundamental domain
p        = {'a', 'b', 'c', 'd', 'mu_s'};    % paramter names
deltamax = 5e-4;
prob     = coco_prob();
prob     = iscrn_settings(prob);
prob     = coco_set(prob, 'cont', 'ItMX', [0, 100]);
prob     = ode_isol2iscrn(prob, 'T0', @tube, @tube_DFDX, @tube_DFDP,...
  sol0.tbp, sol0.xbp, p, sol0.p, sol0.w, [],...
  '-fund', {deltamax; sol0.mu}, '-arc', {0, 10}, '-u');
ISO{1}   = coco(prob, 'q/arc00+', [], 1,...
  {'tau', 'delta', 'x0_1', 'x0_2', 'l'},...
  {[], [-1, 1], [], [], []});

%% Extracting the fundamental domain
%
% The following commands extract the short section of isochron computed while
% finding the fundamental domain of the isochron, and plot this short sgement.
% Also plotted, the change in 'delta' with respect to 'tau' during continuation.
% This data is extracted from the output of the coco entry-point function,
% using post-processing utilities provided by CoCo.

tau   = coco_bd_col(ISO{1}, 'tau');
delta = coco_bd_col(ISO{1}, 'delta');
x0    = coco_bd_col(ISO{1}, {'x0_1', 'x0_2'});
figure(1)
plot(tau, delta)
xlabel('tau')
ylabel('delta')
figure(2)
plot(x0(1,:), x0(2,:))
xlabel('x')
ylabel('y')

%% Compute the first arcs
%
% We use the fundamental domain computed above to construct the return map that
% defines the follwoing arcs of the zero-phase isochron. We set the number of
% mesh points for this computation to 50, and extract the number of mesh points
% used in the fundamental solution to 's_NTST'. For the first arc, the starting
% solution is the same as the solution from the fundamental domain, so 'NTST'
% setting of the 'coll' toolbox is set as such. The 'NTSTMX' setting sets the
% maximum number of mesh points, and is set as 'ntst'. The call to the
% constructor ODE_ISCRN2ISCRN extracts the solution above from the directory and
% object identifer supplied. The following call the to coco entry-point function
% intiates continuation, storing data in the 'q/arc01+' directory. The '-arc'
% option used when computing the fundamental domain is passed along by
% ODE_ISCRN2ISCRN, and 'tau' is scaled such that it varies from 0 to 1 as the
% start point of the isochronal return map moves along the fundamental domain.
%
% The second arc is computed using the terminal solution fo the first arc. The
% constructor ODE_ISCRN2ISCRN concatenates this terminal solution with the
% solution the defines the fundamental domian. This constructs the isochronal
% return map for twice the period of the stable periodic orbit to ehich the
% isochorn belongs. Hence, the number of mesh points for this initial solution
% is set as the addition of 'k_NTST' and 's_NTST', and 'NTSTMX' is set as twice
% 'ntst'. The first three arguments of ODE_ISCRN2ISCRN following prob reference
% the terminal solution fo the first arc, and the next three arguments reference
% the terminal solution of the fundamental arc. The follwoing call to the CoCo
% ebtry-point function intiates continuation.


s_NTST = coco_bd_val(ISO{1}, 2, 'T0.iscrn.NTST');
prob   = coco_prob();
prob   = iscrn_settings(prob);
prob   = coco_set(prob, 'coll', 'NTST', s_NTST, 'NTSTMX', 50);
prob   = ode_iscrn2iscrn(prob, 'T1', 'q/arc00+', 'T0', 2,...
  'q/arc00+', 'T0', 2);
ISO{2} = coco(prob, 'q/arc01+', [], 1, {'tau', 'x0_1', 'x0_2', 'l'},...
  {[-0.1, 1], [-1,1], [-1,1], []});

% compute second arc

k_NTST = coco_bd_val(ISO{2}, 2, 'T1.iscrn.NTST');
prob   = coco_prob();
prob   = iscrn_settings(prob);
prob   = coco_set(prob, 'coll', 'NTST', k_NTST + s_NTST, 'NTSTMX', 100);
prob   = ode_iscrn2iscrn(prob, 'T2', 'q/arc01+', 'T1', 2,...
  'q/arc00+', 'T0', 2);
ISO{3} = coco(prob, 'Gs/arc02+', [], 1, {'tau', 'x0_1', 'x0_2', 'l'},...
  {[-0.1, 1], [-1,1], [-1,1], []});

%% Visualising isochron arcs
%
% The post-processing functions ISCRN_READ_SOLUTION, ISCRN_READ_ARC, and
% ISCRN_READ_ISOCHRON allow for the extraction of data produced by the 'iscrn'
% toolbox constructors. The utility ISCRN_READ_SOLUTION extracts the solution
% to the isochron boundary value problem given a run and label number. The
% ISCRN_READ_ARC utility instead extracts the isochron arc, more specifically
% the continuation variables requested in the cell array taken as its final
% argument. Below we extract the terminal solution of the fundamental arc, and
% the first arc of the zero-phase isochron, and then plot that arc in figure 1.
%
% The utility ISCRN_READ_ISOCHRON reads the arcs of the isochron stored in
% a directory structure formatted accrding to the string provided as its
% first argument. This string format is the same as used by MATLAB's SPRINTF.
% The second string argument is optional; its inclusion adds more data to the
% output strucrue 'iscrn'. Following the cell array of continuation vairbales to
% extract, the option '-trunc' truncates this data to avoid repetiion where the
% arcs of the isochron are joined. The follwoing commands plot the zero-phase
% isochron computed above, as well as plotting the arclength of the isochron
% against its  x-coordiante.

% Isochron arc and solution

sol = iscrn_read_solution('T0', 'q/arc00+', 2);
arc = iscrn_read_arc('q/arc01+', 'T1', {'x0_1', 'x0_2'});

figure(1)
plot(arc.iscrn(:,1), arc.iscrn(:,2))
xlabel('x')
ylabel('y, 'rotation', 0)

% Isochron

iscrn = iscrn_read_isochron('Gs/arc%02d+', 'T%d', {'x0_1', 'x0_2'},...
  '-trunc', 'initial');

figure(2)
plot(iscrn.iscrn(:,1), iscrn.iscrn(:,2), 'b')
hold on
plot(iscrn.xpo(:,1), iscrn.xpo(:,2), 'k')
xlabel('x')
ylabel('y', 'rotation', 0)

figure(3)
plot(iscrn.iscrn(:,1), iscrn.l, 'b')
xlabel('x')
ylabel('l', 'rotation', 0)
