%% Compute a foliation of isochrons
%
% The variables in this section of the script are those that should be
% changed when computing isochrons of a new object, or in a new vector
% field. Particulalrly, make sure that the stability of the oscillation is
% set correctly in the variable stab, and that FolTyp is set as 'ep' for
% equilibrium points or 'po' for periodic orbits. When set to 'ep',
% additionally fill in the fields in the next code section for the
% equilibrium, phase of isochrons desired and radius of the blow up, as x0,
% theta, and epsilon, respectively. 

func      = {@tube, @tube_DFDX, @tube_DFDP};  % functions for vector field
p         = {'a', 'b', 'c', 'd', 'mu_s'};  % paramter names
deltamax  = 5e-4;
prob      = coco_prob();  % get ntst form iscrn_settings
prob      = iscrn_settings(prob);    % get ntst form iscrn_settings
ntst      = coco_get(prob, 'coll');  % get ntst form iscrn_settings
ntst      = ntst.NTST;    % get ntst form iscrn_settings
Narcs     = 5;   % number of arc of isochorn to compute
arc_num   = length(num2str(Narcs));  % length of string of numbers in arcnam
iscrn_nam = 'ep/iscrn';  % name of isochron for stored data
lmax      = 10;  % desired terminal isochron arclength
bounds    = [-1, -1; 1, 1];  % desired computational bounds of isochron, [x,y]
xl        = transpose(bounds(1,:));  % left computational bounds on state variables
xr        = transpose(bounds(2,:));  % right computational bounds on state variables
stab      = '-u';  % stability of object, '-u' if unstable
folTyp    = 'ep';  % choose whether foliation is for periodic orbits: 'po',
% or of an equilibrium: 'ep' - change settings below for 'ep'

%% Choose foliation type, PO or EP.

if strcmp(folTyp, 'ep')
  % EP settings
  x0      = [0.3253; -0.2414];
  p0      = [-0.66; 0.5; 2.5; 2.5; 1.0];
  theta   = 0 : (1/10) : 0.99;
  epsilon = 1e-4;
  Nphase  = numel(theta);
  sol     = cell(Nphase, 1);
  for phi = 1 : Nphase
    sol{phi} = ep_bundle(@tube_DFDX, x0, p0, epsilon, theta(phi));
  end
elseif strcmp(folTyp, 'po')
  THTlabs   = coco_bd_labs(FLQT, 'THETA');
  Nphase    = numel(THTlabs);
  sol       = cell(Nphase, 1);
  for phi = 1 : Nphase
    sol{phi} = flqt_read_solution('po', 'floquet', THTlabs(phi));
  end
end
%% Compute isochron folaition

iscrn_num = length(num2str(Nphase));  % amount of padding in isochorn name, dependant on Nphase
for phi = 1 : Nphase
  isonam  = sprintf('%s%0*d', iscrn_nam, iscrn_num, phi - 1);  
  fprintf('\n%s\n',isonam)
  [~]     = rmdir(sprintf('data/%s', isonam), 's');    % clear old isochron computational data - [~] stops errors when directories are already cleared
  breaker = 0;
  flipper = 0;  
  for side = '-+'
    if strcmp(side, '-') && strcmp(folTyp, 'ep')  % if computing isochrons of an equilibrium
      break  % only comput '-' side
    end
    t0 = sol{phi}.tbp;
    x0 = sol{phi}.xbp;
    p0 = sol{phi}.p;
    mu = sol{phi}.mu;
    w  = str2double([side, '1']) * sol{phi}.w;
    delta_max  = {deltamax; mu};    
    etamax     = abs(mu * sqrt(deltamax));
    arc_l      = {0, lmax};
    arcnam     = sprintf('%s/arc%0*d%s', isonam, arc_num, 0, side);
    fprintf('\n%s\n', arcnam)
    segnam     = sprintf('T%d', 0);

    % fundamental domain       
    % compute fundamental domain
    ISO = cell(Narcs, 1);
    iso = cell(Narcs, 1);
    for j = [1, 2]
      prob = coco_prob();
      prob = iscrn_settings(prob); 
      % get settings for use later
      tol  = coco_get(prob, 'all', 'TOL');
      itmx = coco_get(prob, 'cont', 'ItMX');  % set coco to forward continuation only
      if j == 1
        itmx = [0, max(itmx)];
      else
        itmx = [max(itmx), 0];
      end
      prob = coco_set(prob, 'cont', 'ItMX', itmx);
      prob = ode_isol2iscrn(prob, segnam, func{:}, t0, x0,...
        p, p0, w, [], '-fund', delta_max, '-arc', arc_l, stab);         
      ISO{1} = coco(prob, arcnam, [], 1,...
        {'tau', 'delta', 'x0_1', 'x0_2', 'l'},...
        {[], [ -1, 1 ], bounds(:,1), bounds(:,2), []}); 

      % check correct direction along the eigenvector has been travelled
      EPlabs = coco_bd_labs(ISO{1}, 'EP');
      tauchk = coco_bd_val(ISO{1}, max(EPlabs), 'tau') - ...
        coco_bd_val(ISO{1}, min(EPlabs), 'tau');
      if tauchk > 0
        break
      elseif j == 1
        fprintf('\nComputation has occured towards wrong direction along the linear approximation, re-running\n')
        flipper = 1;
      else
        fprintf('\nComputation has occured towards wrong direction along the linear approximation, unable to correct isochron has been skipped\n')
        breaker = 1;
      end
    end
    if breaker
      fprintf('\nComputation terminated for above reason\n')
      break
    end

    % check that previous solution is valid for restart
    ENDlab = coco_bd_labs(ISO{1});    % if the start point is an MX, not MX or something
    if sum(strcmp(coco_bd_val(ISO{1}, min(ENDlab), 'TYPE'), {'MX'; 'EX'}))
      fprintf('\nComputation start failed\n')
      break
    elseif (strcmp(coco_bd_val(ISO{1}, max(ENDlab), 'TYPE'), 'EP'))
      if numel(EPlabs) == 1    % only the start point exists
        fprintf('\nComputation start failed\n')
        break
      end
      iso{1} = coco_bd_col(ISO{1}, {'tau', 'x0_1', 'x0_2', 'l' }); 
      if flipper    % if continuation has occured 'backwards', flip the data
        iso{1} = fliplr(iso{1});
      end
      if sum((iso{1}(2:3,end) - xl) <= tol, 1) > 0 ||...
           sum((iso{1}(2:3,end) - xr) >= -tol) > 0    % if rectangular computational bounds are exceeded
        fprintf('\nComputation window achieved\n%s%s saved\n', isonam, side)
        iso = transpose([iso{:}]);
        break
      elseif iso{1}(4,end) >= lmax    % if maximum arclength is exceeded
        fprintf('\nMaximum arclength achieved\n%s%s saved\n', isonam, side)
        iso = transpose([iso{:}]);
        break
      end
    end

    % define fundamental domain labels
    fnd_arcnam = arcnam;
    fnd_segnam = segnam;  
    FNDlab     = max(EPlabs);
    EPlab      = FNDlab;  % restart label for first arc of isochron
    s_NTST     = coco_bd_val(ISO{1}, FNDlab, sprintf('%s.iscrn.NTST', segnam)) ;
    k_NTST     = 0;  % don't add mesh points to initial continuation along fundamental domain
    % iterate over segments
    for k = 1:Narcs 
      prob    = coco_prob();
      prob    = iscrn_settings(prob);
      prob    = coco_set(prob, 'coll', 'NTST', k_NTST + s_NTST,...
        'NTSTMX', k * ntst);      
      segnam_ = segnam; 
      segnam  = sprintf('T%d', k);
      prob    = ode_iscrn2iscrn(prob, segnam, arcnam, segnam_, EPlab,...
        fnd_arcnam, fnd_segnam, FNDlab);
      arcnam  = sprintf('%s/arc%0*d%s', isonam, arc_num, k, side);
      fprintf('\n%s\n',arcnam) 

      ISO{k+1} = coco(prob, arcnam, [], 1,{'tau', 'x0_1', 'x0_2', 'l'},... 
        {[-0.1, 1], bounds(:,1), bounds(:,2), []});

      % check that previous solution is valid for restart
      ENDlab = coco_bd_labs(ISO{k+1});    
      % if the start point is an MX, MXCL, or EX - terminate computation
      if sum(strcmp(coco_bd_val(ISO{k+1}, min(ENDlab), 'TYPE'),...
           {'MX'; 'MXCL'; 'EX'}))
        fprintf('\nComputation start failed\n%s%s saved\n', isonam, side)
        break
      % if the end point is an EP - expected for normal circumstances
      elseif (strcmp( coco_bd_val(ISO{k+1}, max(ENDlab), 'TYPE' ), 'EP'))
        EPlabs  = coco_bd_labs(ISO{k+1}, 'EP');
        % only the start point exists - terminate computation
        if numel(EPlabs) == 1    
          fprintf('\nComputation start failed\n%s%s saved\n', isonam, side)
        end
        iso{k+1} = coco_bd_col(ISO{k+1}, {'tau', 'x0_1', 'x0_2', 'l'}); 
        % if rectangular computational bounds are met or exceeded - terminate computation
        if sum((iso{k+1}(2:3,end) - xl) <= tol ) > 0 ||...
             sum((iso{k+1}(2:3,end) - xr) >= -tol) > 0    
          fprintf('\nComputation window achieved\n%s%s saved\n', isonam, side)
          break
        % if maximum arclength is exceeded - terminate computation
        elseif iso{k+1}(4,end) >= lmax    
          fprintf('\nMaximum arclength achieved\n%s%s saved\n', isonam, side)
          break
        end        
      % if maximum arclength is exceeded - terminate computation
      elseif (strcmp(coco_bd_val(ISO{k+1}, max(ENDlab), 'TYPE'), 'ARC'))    
        fprintf('\nMaximum arclength achieved\n%s%s saved\n', isonam, side)
        break
      else
        fprintf('\nComputation failed to end\n%s%s saved\n', isonam, side)
        break
      end 
      EPlab  = max(ENDlab);
      k_NTST = coco_bd_val(ISO{k+1}, EPlab, sprintf( '%s.iscrn.NTST', segnam));
    end
  end  
end